try {
    require('electron-reloader')(module)
} catch (_) {}

const { app, BrowserWindow, session, shell, ipcMain, Tray, Menu, nativeImage } = require("electron");
const path = require("path");
const { spawn, exec } = require("child_process");
const fs = require("fs");
const fsPromises = require("fs").promises;

// ========================================
// ✅ COMPLETE CACHE & SERVICE WORKER FIX
// ========================================

// Set custom cache directory with proper permissions
app.setPath('userData', path.join(app.getPath('appData'), 'Naroapp'));
app.setPath('cache', path.join(app.getPath('userData'), 'cache'));

// Disable problematic caching features
app.commandLine.appendSwitch('disable-http-cache');
app.commandLine.appendSwitch('disable-gpu-shader-disk-cache');

// Disable service workers to prevent database errors
app.commandLine.appendSwitch('disable-features', 'ServiceWorkerOnUI');
app.commandLine.appendSwitch('disable-background-networking');

// Windows-specific fixes
if (process.platform === 'win32') {
    app.commandLine.appendSwitch('disable-software-rasterizer');
    // Prevent permission issues
    app.commandLine.appendSwitch('no-sandbox');
}

// ========================================
// ✅ END FIX SECTION
// ========================================

let win;
let tray = null;
let appSettings = {
  popupBeforeInstall: true,
  autoInstall: false,
  runAsAdmin: false,
  startupLaunch: false,
  minimizeToTray: false,
  ecoMode: false,
  theme: 'default',
  syncEnabled: false  // ✅ ADD THIS
};

// Global Map to track active DownloadItems by filename
const activeDownloads = new Map();

// Persistent storage paths
// Find around line 30:
const userDataPath = app.getPath('userData');

// ✅ ADD THESE LINES:
const internalDataPath = path.join(userDataPath, 'internal-data');
const downloadsDataPath = path.join(internalDataPath, 'downloads');
const appsDataPath = path.join(internalDataPath, 'apps');
const settingsDataPath = path.join(internalDataPath, 'settings');

// Ensure directories exist
[internalDataPath, downloadsDataPath, appsDataPath, settingsDataPath].forEach(dir => {
    if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
    }
});
const profilesPath = path.join(userDataPath, 'profiles');
const currentUserPath = path.join(userDataPath, 'current-user.json');

// Ensure profiles directory exists
if (!fs.existsSync(profilesPath)) {
    fs.mkdirSync(profilesPath, { recursive: true });
}
const firebaseConfig = {
  apiKey: "AIzaSyB3mXKarHWr4ZjEdvWVfJi4wIFni7ckbe0",
  authDomain: "naroapp-webstore.firebaseapp.com",
  databaseURL: "https://naroapp-webstore-default-rtdb.asia-southeast1.firebasedatabase.app",
  projectId: "naroapp-webstore",
  storageBucket: "naroapp-webstore.firebasestorage.app",
  messagingSenderId: "1000553964526",
  appId: "1:1000553964526:web:d8aedcc77276112e46a4ff"
};

// Get current user identifier
function getCurrentUserId() {
    try {
        if (fs.existsSync(currentUserPath)) {
            const data = JSON.parse(fs.readFileSync(currentUserPath, 'utf8'));
            return data.uid || 'guest';
        }
    } catch (e) {
        console.error("Error reading current user:", e);
    }
    return 'guest';
}

// Get user-specific paths
function getUserPaths(userId) {
    const userDir = path.join(profilesPath, userId);
    if (!fs.existsSync(userDir)) {
        fs.mkdirSync(userDir, { recursive: true });
    }
    return {
        userDir: userDir,
        profile: path.join(userDir, 'profile.json'),
        downloads: path.join(userDir, 'downloads.json'),
        settings: path.join(userDir, 'settings.json')
    };
}

// Load settings from disk on startup
function loadAppSettings() {
    try {
        const userId = getCurrentUserId();
        const paths = getUserPaths(userId);
        
        if (fs.existsSync(paths.settings)) {
            const data = fs.readFileSync(paths.settings, 'utf8');
            appSettings = { ...appSettings, ...JSON.parse(data) };
        }
    } catch (e) {
        console.error("Error loading app settings:", e);
    }
}

// Save settings to disk
function saveAppSettings(userId = null) {
    try {
        const uid = userId || getCurrentUserId();
        const paths = getUserPaths(uid);
        fs.writeFileSync(paths.settings, JSON.stringify(appSettings, null, 2));
    } catch (e) {
        console.error("Error saving app settings:", e);
    }
}

// Windows Startup Registry Management
function setWindowsStartup(enable) {
    if (process.platform !== 'win32') return;
    
    const appPath = app.getPath('exe');
    const regKey = 'HKCU\\Software\\Microsoft\\Windows\\CurrentVersion\\Run';
    const regName = 'Naroapp';
    
    if (enable) {
        exec(`reg add "${regKey}" /v "${regName}" /t REG_SZ /d "${appPath}" /f`, (error) => {
            if (error) console.error('Failed to add startup entry:', error);
        });
    } else {
        exec(`reg delete "${regKey}" /v "${regName}" /f`, (error) => {
            if (error) console.error('Failed to remove startup entry:', error);
        });
    }
}

// Run as Admin on next launch
function setRunAsAdmin(enable) {
    if (process.platform !== 'win32') return;
    
    const appPath = app.getPath('exe');
    const shortcutPath = path.join(app.getPath('appData'), 'Microsoft', 'Windows', 'Start Menu', 'Programs', 'Naroapp.lnk');
    
    if (enable) {
        // Create a shortcut with elevated privileges
        const script = `
$WshShell = New-Object -comObject WScript.Shell
$Shortcut = $WshShell.CreateShortcut("${shortcutPath}")
$Shortcut.TargetPath = "${appPath}"
$Shortcut.Save()
        `;
        const psPath = path.join(userDataPath, 'create-shortcut.ps1');
        fs.writeFileSync(psPath, script);
        exec(`powershell -ExecutionPolicy Bypass -File "${psPath}"`, (error) => {
            if (error) console.error('Failed to create admin shortcut:', error);
        });
    }
}

// Create System Tray
function createTray() {
    const iconPath = path.join(__dirname, 'icon.png');
    let trayIcon;
    
    try {
        trayIcon = nativeImage.createFromPath(iconPath);
        if (trayIcon.isEmpty()) {
            trayIcon = nativeImage.createEmpty();
        } else {
            trayIcon = trayIcon.resize({ width: 16, height: 16 });
        }
    } catch (e) {
        trayIcon = nativeImage.createEmpty();
    }
    
    tray = new Tray(trayIcon);
    
    const contextMenu = Menu.buildFromTemplate([
        {
            label: 'Show Naroapp',
            click: () => {
                if (win) {
                    win.show();
                    win.focus();
                }
            }
        },
        {
            label: 'Hide',
            click: () => {
                if (win) win.hide();
            }
        },
        { type: 'separator' },
        {
            label: 'Quit',
            click: () => {
                app.isQuitting = true;
                app.quit();
            }
        }
    ]);
    
    tray.setToolTip('Naroapp');
    tray.setContextMenu(contextMenu);
    
    tray.on('click', () => {
        if (win) {
            if (win.isVisible()) {
                win.hide();
            } else {
                win.show();
                win.focus();
            }
        }
    });
}

function createWindow() {
  win = new BrowserWindow({
    show: false,
    backgroundColor: '#0a0a0a',
    icon: path.join(__dirname, "icon.ico"),
    autoHideMenuBar: true,
    frame: false,
    titleBarStyle: 'hidden',
    webPreferences: {
      preload: path.join(__dirname, "preload.js"),
      webviewTag: true,
      contextIsolation: true,
      sandbox: false,
      nodeIntegration: false,
      backgroundThrottling: false
    }
  });


  win.maximize();
  
  win.webContents.setWindowOpenHandler(() => {
    return { action: 'deny' };
  });

  win.once('ready-to-show', () => {
    win.show();
  });

  loadAppSettings(); // Ensure settings loaded
  
  // Determine which HTML to load
  const syncEnabled = appSettings.syncEnabled || false;
  win.loadFile("index.html");
  console.log(`🔄 Loaded index.html`);

  win.webContents.once('did-finish-load', () => {
  const webviewSession = session.fromPartition('persist:main');
  // Remove any existing listeners to avoid duplicates
  webviewSession.removeAllListeners('will-download');
  setupDownloadHandler(webviewSession);
  console.log('✅ Download handler re-attached after window load');
});

  win.on('maximize', () => {
    win.webContents.send('window-state-change', { isMaximized: true });
  });

  win.on('unmaximize', () => {
    win.webContents.send('window-state-change', { isMaximized: false });
  });

  // Handle window close - minimize to tray if enabled
  win.on('close', (event) => {
    if (!app.isQuitting && appSettings.minimizeToTray) {
      event.preventDefault();
      win.hide();
      return false;
    }
  });


} // END of createWindow

// ====================================================================
// ✅ DOWNLOAD TRACKING FIX: Moved setupDownloadHandler outside of createWindow
// ====================================================================

// --- DOWNLOAD HANDLER LOGIC (Moved into a reusable function) ---
function setupDownloadHandler(targetSession) {
    const partition = targetSession.partition || 'default';
    console.log(`🔧 [MAIN] Setting up download handler for session: ${partition}`);
    
    // Remove existing listeners to prevent duplicates
    targetSession.removeAllListeners('will-download');
    
    targetSession.on("will-download", (event, item, webContents) => {
        console.log('\n🚨 ========== DOWNLOAD INTERCEPTED ========== 🚨');
        console.log(`📦 Filename: ${item.getFilename()}`);
        console.log(`📏 Size: ${(item.getTotalBytes() / 1024 / 1024).toFixed(2)} MB`);
        console.log(`🔗 URL: ${item.getURL()}`);
        console.log(`🪟 Window exists: ${!!win}`);
        console.log(`📡 WebContents ready: ${!!win?.webContents}`);
        console.log('============================================\n');
        
        const filename = item.getFilename();
        const totalBytes = item.getTotalBytes();
        
        // Store item for control (pause/resume/cancel)
        activeDownloads.set(filename, item);

        if (!win || !win.webContents) {
            console.error("❌ [MAIN] Window not ready - cancelling download");
            item.cancel();
            activeDownloads.delete(filename);
            return;
        }
        
        // Let native "Save As" dialog appear (DON'T set save path)
        console.log('📨 [MAIN] Sending download-start to renderer...');
        const sendResult = win.webContents.send("download-start", { filename, totalBytes });
        console.log('📨 [MAIN] Send result:', sendResult);
        
        item.on("updated", (event, state) => {
            if (state === 'progressing') {
                const received = item.getReceivedBytes();
                const progress = totalBytes > 0 ? Math.round((received / totalBytes) * 100) : -1;
                
                // Log every 25% progress
                if (progress % 25 === 0) {
                    console.log(`📊 [MAIN] Download progress: ${filename} - ${progress}%`);
                }
                
                if (win && !win.isDestroyed()) {
                    win.setProgressBar(progress >= 0 ? progress / 100 : -1);
                    win.webContents.send("download-progress", { filename, progress, received, totalBytes, state: 'progressing' });
                }
            } else if (state === 'interrupted') {
                console.log(`⏸️ [MAIN] Download interrupted/paused: ${filename}`);
                if (win && !win.isDestroyed()) {
                    win.webContents.send("download-paused", { filename });
                }
            }
        });

        item.on("done", (event, state) => {
            console.log(`\n✅ [MAIN] Download done: ${filename} - State: ${state}`);
            // Remove from active map
            activeDownloads.delete(filename);

            if (win && !win.isDestroyed()) win.setProgressBar(-1);
            
            const savedPath = item.getSavePath();
            const totalSize = (item.getTotalBytes() / (1024 * 1024)).toFixed(2) + ' MB';
            
            console.log(`💾 [MAIN] Saved to: ${savedPath}`);
            console.log(`📨 [MAIN] Sending download-complete to renderer...`);

            if (state === "completed") {
                win.webContents.send("download-complete", { 
                    filename, 
                    path: savedPath, 
                    success: true, 
                    totalSize 
                });

                // Check settings for auto-install/popup
                const currentSettings = appSettings; 
                
                if (currentSettings.autoInstall === true) {
                    console.log('🔧 [MAIN] Auto-install triggered');
                    setTimeout(() => {
                        installAppLogic(savedPath);
                    }, 500);
                } else if (currentSettings.popupBeforeInstall === true) {
                    console.log('📋 [MAIN] Showing install modal');
                    setTimeout(() => {
                        win.webContents.send("show-install-modal", { filename, path: savedPath });
                    }, 500);
                }
            } else {
                // Cancelled or Interrupted final state
                win.webContents.send("download-complete", { 
                    filename, 
                    path: savedPath, 
                    success: false, 
                    totalSize,
                    state: state
                });
            }
            
            console.log('============================================\n');
        });
    });
    
    console.log(`✅ [MAIN] Download handler attached to ${partition} session\n`);
}
// ====================================================================
// ✅ END DOWNLOAD TRACKING FIX
// ====================================================================

// --- IPC: DOWNLOAD CONTROLS ---
ipcMain.on('pause-download', (event, filename) => {
    const item = activeDownloads.get(filename);
    if (item && !item.isPaused()) {
        item.pause();
        if (win) win.webContents.send("download-paused", { filename });
    }
});

ipcMain.on('resume-download', (event, filename) => {
    const item = activeDownloads.get(filename);
    if (item && item.canResume()) {
        item.resume();
        if (win) win.webContents.send("download-resumed", { filename });
    }
});

ipcMain.on('cancel-download', (event, filename) => {
    const item = activeDownloads.get(filename);
    if (item) {
        item.cancel();
        // The 'done' event will fire with state 'cancelled'
    }
});

// --- HELPER: Safe Install Logic ---
function installAppLogic(filePath) {
    if (!filePath || !fs.existsSync(filePath)) return;

    try {
        const stat = fs.statSync(filePath);
        if (stat.isDirectory()) {
            shell.openPath(filePath);
            return;
        }
    } catch(e) { return; }

    const ext = path.extname(filePath).toLowerCase();
    
    if (ext === '.msi') {
        const child = spawn("msiexec", ["/i", filePath, "/passive"], { stdio: 'ignore', windowsHide: false });
        child.on('error', (e) => console.error("MSI Install Error:", e));
    } else {
        shell.openPath(filePath).catch(e => console.error("EXE Launch Error (Shell):", e));
    }
}

app.on("web-contents-created", (event, contents) => {
  if (contents.getType() === "webview") {
    contents.setWindowOpenHandler(({ url }) => {
      console.log('🔗 Main process: New window requested:', url);
      
      // Send to renderer to open in new tab
      if (win && win.webContents) {
        win.webContents.send('open-url-in-new-tab', url);
      }
      
      // Deny the actual window creation (we'll handle it in renderer)
      return { action: 'deny' };
    });
  }
});

// --- ASYNC HELPER: DEEP SEARCH FOR APPS (NON-BLOCKING) ---
async function findAppFolderRecursiveAsync(basePath, searchName, currentDepth = 0, maxDepth = 5) {
    if (currentDepth > maxDepth) return null;
    
    try {
        await fsPromises.access(basePath);
    } catch { return null; }

    try {
        const entries = await fsPromises.readdir(basePath, { withFileTypes: true });
        
        for (const entry of entries) {
            if (entry.isDirectory()) {
                const folderName = entry.name.toLowerCase();
                const cleanFolder = folderName.replace(/\s/g, '');
                const cleanSearch = searchName.replace(/\s/g, '');
                
                if (cleanFolder.includes(cleanSearch) || folderName.includes(cleanSearch)) {
                    return path.join(basePath, entry.name);
                }
                
                const foundInSub = await findAppFolderRecursiveAsync(
                    path.join(basePath, entry.name), 
                    searchName, 
                    currentDepth + 1, 
                    maxDepth
                );
                
                if (foundInSub) return foundInSub;
            }
        }
    } catch (e) { return null; }
    return null;
}

// ====================================================================
// ✅ APP DETECTION FIX: Improved AI Scoring & Filtering
// ====================================================================

async function findMainExecutableAsync(folderPath, appName) {
    try {
        await fsPromises.access(folderPath);
        const files = await fsPromises.readdir(folderPath);
        
        let exeFiles = files.filter(file => file.toLowerCase().endsWith('.exe'));
        
        // 1. CRITICAL: Handle folder path if it IS the executable
        if (exeFiles.length === 0) {
            // Check if the folderPath itself points directly to an EXE.
            if (folderPath.toLowerCase().endsWith('.exe') && await isFile(folderPath)) {
                return folderPath;
            }
            return null; 
        }

        const searchName = appName.toLowerCase().replace(/\s+/g, '');
        const folderName = path.basename(folderPath).toLowerCase().replace(/\s+/g, '');
        
        // 🤖 ULTRA-SMART AI EXCLUSION LIST (Preserved and kept comprehensive)
        const exclusions = [
            'uninstall', 'uninst', 'unins', 'setup', 'install', 'installer', 'update', 'updater', 
            'patch', 'patcher', 'migrate', 'migration', 'bootstrapper', 'stub', 'downloader', 'loader',
            'bugreport', 'bug report', 'bugsplat', 'crash', 'crashreport', 'crashhandler', 
            'errorreport', 'error report', 'report', 'reporter', 'diagnostic', 'diagnostics', 
            'feedback', 'telemetry', 'dump', 'drwatson', 'werfault', 'bug report tool',
            'helper', 'service', 'launcher', 'watchdog', 'daemon', 'monitor', 'logger',
            'background', 'agent', 'tray', 'notify', 'notification', 'notifier', 'toast', // Added toast
            'client', 'webclient', 'network', 'web', 'browser', 'online', 'webhelper',
            'utility', 'util', 'tool', 'manager', 'config', 'configuration', 'setting', 
            'settings', 'control', 'console', 'cmd', 'terminal', 'cli', // Added cli
            'performance', 'monitor', 'watcher', 'checker', 'validator', 'verify', 'verification',
            'cache', 'temp', 'tmp', 'cleanup', 'cleaner', 'clear',
            'bugreporttool', 'installmanager', 'ccc', 'cnext', 'cnext64', 'nvcontainer',
            'nvcplui', 'nvprofileupdater', 'geforce', 'shadowplay', 'share', 'nvbackend',
            'unwise', 'isuninst', 'uninst', '_uninstall', 'del', 'remove', 'remover',
            'editor', // ✅ NEW: often an internal tool, not the main launch point
            'gameoverlay', // ✅ NEW: common helper tool
        ];
        
        const isFile = async (filePath) => {
            try {
                const stat = await fsPromises.stat(filePath);
                return stat.isFile();
            } catch (e) {
                return false;
            }
        };

        // 🎯 SMART FILTER PHASE 1: Remove obvious non-apps
        let cleanExeFiles = exeFiles.filter(file => {
            const name = file.toLowerCase().replace('.exe', '');
            const cleanName = name.replace(/[^a-z0-9]/g, '');
            
            // Check exclusions
            const isExcluded = exclusions.some(exclusion => {
                const cleanExclusion = exclusion.replace(/[^a-z0-9]/g, '');
                return cleanName.includes(cleanExclusion) || name.includes(exclusion);
            });
            
            return !isExcluded;
        });
        
        // Fallback if all filtered
        if (cleanExeFiles.length === 0) {
            console.log(`⚠️ All executables filtered for ${appName}, using original list`);
            cleanExeFiles = exeFiles;
        }

        // 🧠 AI SCORING SYSTEM - Score each executable
        const scoredFiles = await Promise.all(cleanExeFiles.map(async file => {
            let score = 0;
            const fileName = file.toLowerCase().replace('.exe', '');
            const cleanFile = fileName.replace(/[^a-z0-9]/g, '');
            
            // Score 1: Exact name match (highest priority)
            if (cleanFile === searchName) score += 1000;
            
            // Score 2: Contains app name
            if (cleanFile.includes(searchName)) score += 500;
            if (searchName.includes(cleanFile) && cleanFile.length > 3) score += 400;
            
            // Score 3: Folder name match
            if (cleanFile === folderName) score += 300;
            if (folderName.includes(cleanFile)) score += 200;
            
            // Score 4: File size (main apps are usually larger)
            try {
                const fullPath = path.join(folderPath, file);
                const stats = await fsPromises.stat(fullPath);
                const sizeMB = stats.size / (1024 * 1024);
                
                if (sizeMB > 50) score += 150;  // Large files likely main app
                else if (sizeMB > 10) score += 100;
                else if (sizeMB > 1) score += 50;
                else if (sizeMB < 0.5) score -= 100; // Very small files likely helpers
            } catch(e) {}
            
            // Score 5: Naming patterns that indicate main app
            if (fileName.includes('main')) score += 80;
            if (fileName.includes('app')) score += 70;
            if (fileName.includes('client') && !fileName.includes('web')) score += 60;
            if (fileName.includes('game')) score += 90;
            if (fileName.includes('player')) score += 70;
            if (fileName.includes('studio')) score += 80;
            
            // Score 6: Version numbers (apps often have clean names)
            const hasVersionPattern = /\d+\.\d+/.test(fileName);
            if (hasVersionPattern) score -= 50; // Versioned files often utilities
            
            // Score 7: Underscore/dash heavy names (often utilities)
            const specialCharCount = (fileName.match(/[_-]/g) || []).length;
            if (specialCharCount > 2) score -= 30;
            
            // Score 8: Single word names (often main apps)
            const wordCount = fileName.split(/[_-\s]/).length;
            if (wordCount === 1 && fileName.length > 3) score += 100;
            
            return { file, score };
        }));

        // 🏆 Sort by score and pick winner
        scoredFiles.sort((a, b) => b.score - a.score);
        
        // Log top 3 candidates for debugging
        console.log(`🔍 Top candidates for "${appName}":`);
        scoredFiles.slice(0, 3).forEach((item, i) => {
            console.log(`  ${i + 1}. ${item.file} (Score: ${item.score})`);
        });
        
        const bestMatch = scoredFiles[0];
        
        if (bestMatch && bestMatch.score > 0) {
            const finalPath = path.join(folderPath, bestMatch.file);
            console.log(`✅ Selected: ${bestMatch.file} (Score: ${bestMatch.score})`);
            return finalPath;
        }
        
        // Fallback: Check subdirectories (last resort)
        for (const file of files) {
            const subPath = path.join(folderPath, file);
            try {
                const stat = await fsPromises.lstat(subPath);
                if (stat.isDirectory()) {
                    const subFiles = await fsPromises.readdir(subPath);
                    const nestedExe = subFiles.find(f => f.toLowerCase().endsWith('.exe'));
                    if (nestedExe) {
                        const nestedPath = path.join(subPath, nestedExe);
                        console.log(`✅ Found in subdirectory (Fallback): ${nestedExe}`);
                        return nestedPath;
                    }
                }
            } catch(e) {}
        }
        
        console.log(`❌ No suitable executable found for ${appName}`);
        return null;
    } catch (e) { 
        console.log(`❌ Error finding executable for ${appName}:`, e.message);
        return null; 
    }
}
// ====================================================================
// ✅ END APP DETECTION FIX
// ====================================================================

async function findPathInRegistry(appName) {
    if (process.platform !== 'win32') return null;
    const queries = [
        `reg query "HKLM\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall" /s /f "${appName}"`,
        `reg query "HKLM\\SOFTWARE\\WOW6432Node\\Microsoft\\Windows\\CurrentVersion\\Uninstall" /s /f "${appName}"`,
        `reg query "HKCU\\Software\\Microsoft\\Windows\\CurrentVersion\\Uninstall" /s /f "${appName}"`
    ];
    for (const query of queries) {
        try {
            const { stdout } = await new Promise((resolve, reject) => {
                exec(query, (error, stdout, stderr) => {
                    if (error) reject(error);
                    else resolve({ stdout });
                });
            });
            const lines = stdout.split('\r\n');
            let installLocation = null;
             const locLine = lines.find(l => l.includes("InstallLocation"));
             if (locLine) {
                 const parts = locLine.split("REG_SZ");
                 if (parts.length > 1) installLocation = parts[1].trim();
             }
             if (installLocation && fs.existsSync(installLocation)) return installLocation;
        } catch (e) {}
    }
    return null;
}

// --- IPC HANDLERS ---

ipcMain.handle("get-firebase-config", () => firebaseConfig);

// User Profile Management
ipcMain.removeAllListeners("save-user-profile");
ipcMain.on("save-user-profile", (event, profileData) => {
    try { 
        if (profileData && profileData.uid) {
            // Save current user reference
            fs.writeFileSync(currentUserPath, JSON.stringify({ uid: profileData.uid }, null, 2));
            
            // Save profile data
            const paths = getUserPaths(profileData.uid);
            fs.writeFileSync(paths.profile, JSON.stringify(profileData, null, 2));
            
            // Load settings for this user
            loadAppSettings();
        }
    } catch (e) {
        console.error("Error saving user profile:", e);
    }
});

ipcMain.removeAllListeners("clear-user-profile");
ipcMain.handle("clear-user-profile", async (event) => {
    try { 
        console.log('Sign out requested');
        
        if (fs.existsSync(currentUserPath)) {
            fs.unlinkSync(currentUserPath);
        }
        
        appSettings = {
            popupBeforeInstall: true,
            autoInstall: false,
            runAsAdmin: false,
            startupLaunch: false,
            minimizeToTray: false,
            ecoMode: false,
            theme: 'default'
        };
        
        saveAppSettings('guest');
        return { success: true };
    } catch (e) {
        console.error("Error clearing user profile:", e);
        return { success: false, error: e.message };
    }
});

// Get current user ID handler
ipcMain.handle("get-current-user-id", () => {
    return getCurrentUserId();
});

ipcMain.handle("get-user-profile-data", async (event) => {
    try {
        const userId = getCurrentUserId();
        const paths = getUserPaths(userId);
        
        if (fs.existsSync(paths.profile)) {
            return JSON.parse(fs.readFileSync(paths.profile, 'utf8'));
        }
    } catch(e) {
        console.error("Error reading user profile:", e);
    }
    return null; 
});

// Download History Management
ipcMain.handle("get-download-history", async (event) => {
    try {
        const userId = getCurrentUserId();
        const paths = getUserPaths(userId);
        
        if (fs.existsSync(paths.downloads)) {
            return JSON.parse(fs.readFileSync(paths.downloads, 'utf8'));
        }
    } catch(e) {
        console.error("Error reading download history:", e);
    }
    return [];
});

ipcMain.on("save-download-history", (event, historyData) => {
    try {
        if (historyData) {
            const userId = getCurrentUserId();
            const paths = getUserPaths(userId);
            fs.writeFileSync(paths.downloads, JSON.stringify(historyData, null, 2));
            console.log(`[DEBUG MAIN] History saved to disk. Items: ${historyData.length}`); // 👈 ADD THIS
        }
    } catch (e) {
        console.error("Error saving download history:", e);
    }
});

ipcMain.on("clear-download-history", (event) => {
    try {
        const userId = getCurrentUserId();
        const paths = getUserPaths(userId);
        if (fs.existsSync(paths.downloads)) fs.unlinkSync(paths.downloads);
    } catch (e) {
        console.error("Error clearing download history:", e);
    }
});

// Clear all app data for current user
ipcMain.handle("clear-app-data", async (event) => {
    try {
        const userId = getCurrentUserId();
        const paths = getUserPaths(userId);
        if (fs.existsSync(paths.profile)) fs.unlinkSync(paths.profile);
        if (fs.existsSync(paths.downloads)) fs.unlinkSync(paths.downloads);
        if (fs.existsSync(paths.settings)) fs.unlinkSync(paths.settings);
        
        appSettings = {
            popupBeforeInstall: true,
            autoInstall: false,
            runAsAdmin: false,
            startupLaunch: false,
            minimizeToTray: false,
            ecoMode: false,
            theme: 'default'
        };
        
        return { success: true };
    } catch (e) {
        console.error("Error clearing app data:", e);
        return { success: false, error: e.message };
    }
});

// Window Controls
ipcMain.removeAllListeners("control-window");
ipcMain.on("control-window", (event, action) => {
    if (!win) return;
    switch(action) {
        case 'minimize': win.minimize(); break;
        case 'maximize': win.isMaximized() ? win.unmaximize() : win.maximize(); break;
        case 'close': 
            if (appSettings.minimizeToTray) {
                win.hide();
            } else {
                win.close();
            }
            break;
    }
});

ipcMain.handle("get-initial-window-state", () => {
    if (!win) return { isMaximized: false };
    return { isMaximized: win.isMaximized() };
});

ipcMain.handle("get-app-info-data", async (event) => {
    try { 
        return await event.sender.executeJavaScript(`window.appAPI.getAppInfo()`); 
    } catch (e) { 
        return { appName: "Error", publisher: "N/A" }; 
    }
});

// App Settings with System Integration
ipcMain.removeAllListeners("update-app-settings");
ipcMain.on("update-app-settings", (event, settings) => { 
    const oldSettings = { ...appSettings };
    appSettings = { ...appSettings, ...settings };
    
    // Handle system-level settings
    if (settings.startupLaunch !== undefined && settings.startupLaunch !== oldSettings.startupLaunch) {
        setWindowsStartup(settings.startupLaunch);
    }
    
    if (settings.runAsAdmin !== undefined && settings.runAsAdmin !== oldSettings.runAsAdmin) {
        setRunAsAdmin(settings.runAsAdmin);
    }
    
    saveAppSettings();
});

ipcMain.handle("get-app-settings", async () => {
    return appSettings;
});

// File Operations
ipcMain.removeAllListeners("open-file-location");
ipcMain.on("open-file-location", (event, filePath) => {
    if (fs.existsSync(filePath)) shell.showItemInFolder(filePath); 
});

ipcMain.removeAllListeners("delete-file");
ipcMain.on("delete-file", (event, filePath) => {
    try {
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
        event.sender.send("download-file-deleted", { success: true, path: filePath });
    } catch (e) {
        event.sender.send("download-file-deleted", { success: false, path: filePath });
    }
});

ipcMain.removeAllListeners("toggle-dns");
ipcMain.on("toggle-dns", (event, enable) => {
  const command = enable ? `netsh interface ip set dns "Wi-Fi" static 1.1.1.1` : `netsh interface ip set dns "Wi-Fi" dhcp`;
  require("child_process").exec(command);
});

ipcMain.removeAllListeners("uninstall-app");
ipcMain.on("uninstall-app", async (event, appName) => {
    const psCommand = `
    $name = "*${appName}*";
    $paths = @("HKLM:\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall", "HKLM:\\SOFTWARE\\WOW6432Node\\Microsoft\\Windows\\CurrentVersion\\Uninstall", "HKCU:\\Software\\Microsoft\\Windows\\CurrentVersion\\Uninstall");
    $found = $false;
    foreach ($path in $paths) {
        $app = Get-ItemProperty "$path\\*" -ErrorAction SilentlyContinue | Where-Object { $_.DisplayName -like $name } | Select-Object -First 1 UninstallString;
        if ($app) {
             $found = $true;
             $cmd = $app.UninstallString;
             Start-Process cmd -ArgumentList "/c $cmd" -Verb RunAs -WindowStyle Hidden
             break;
        }
    }
    if (-not $found) { Start-Process "ms-settings:appsfeatures" }
    `;
    spawn("powershell.exe", ["-NoProfile", "-ExecutionPolicy", "Bypass", "-Command", psCommand], { stdio: 'ignore', windowsHide: true });
});

ipcMain.removeHandler("open-app");
ipcMain.handle("open-app", async (event, appName) => {
    if (!appName) return false;
    const searchName = appName.toLowerCase().trim();
    const commonPaths = [
        process.env.ProgramFiles,
        process.env["ProgramFiles(x86)"],
        path.join(app.getPath('home'), 'AppData', 'Local', 'Programs'),
        path.join(app.getPath('home'), 'AppData', 'Local'),
        path.join(app.getPath('home'), 'AppData', 'Local', 'Packages') 
    ];

    let appPath = null;
    const regPath = await findPathInRegistry(appName);
    if (regPath) {
         // Using Async Finder here
         appPath = await findMainExecutableAsync(regPath, searchName);
    }

    if (!appPath) {
        for (const basePath of commonPaths) {
            if (!basePath) continue;
            // Using Async Finder here
            const foundFolder = await findAppFolderRecursiveAsync(basePath, searchName);
            if (foundFolder) {
                appPath = await findMainExecutableAsync(foundFolder, searchName);
                if (appPath) break; 
            }
        }
    }

    if (appPath) {
        console.log("Launching app at:", appPath);
        shell.openPath(appPath).catch(error => {
            console.error("Launch Error:", error);
        });
        return true;
    } else {
        return false;
    }
});

// --- UPDATED IPC HANDLER FOR APP SEARCH WITH AGGRESSION LEVELS ---
ipcMain.removeHandler("find-app-path");
ipcMain.handle("find-app-path", async (event, appName, aggressionLevel = 5) => {
    if (!appName) return null;
    
    // ✅ ADD: Validation and logging
    const level = parseInt(aggressionLevel) || 5;
    console.log(`🔍 [Main] Finding app: "${appName}" | Aggression: ${level}`);
    
    const searchName = appName.toLowerCase().trim();
    const searchVariants = [
        searchName,
        searchName.replace(/\s+/g, ''),
        searchName.replace(/\s+/g, '-'),
        searchName.split(' ')[0], 
    ];
    
    console.log(`🔍 Search: ${appName} | Level: ${level}`);

    // LEVEL 1: Registry Only (Fastest)
    if (level >= 1) {
        // Registry Logic
        const registryPaths = [
            'HKLM\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall',
            'HKLM\\SOFTWARE\\WOW6432Node\\Microsoft\\Windows\\CurrentVersion\\Uninstall',
            'HKCU\\Software\\Microsoft\\Windows\\CurrentVersion\\Uninstall',
        ];
        
        for (const regPath of registryPaths) {
            try {
                const { stdout } = await new Promise((resolve) => {
                    exec(`reg query "${regPath}" /s /f "${appName}" 2>nul`, (err, out) => resolve({ stdout: out || '' }));
                });
                if (stdout) {
                    const installMatch = stdout.match(/InstallLocation\s+REG_SZ\s+(.+)/);
                    if (installMatch) {
                        const location = installMatch[1].trim();
                        if (fs.existsSync(location)) {
                            // Use Async Executable finder
                            const exePath = await findMainExecutableAsync(location, searchName);
                            if (exePath) return exePath;
                        }
                    }
                }
            } catch (e) {}
        }
    }

    // LEVEL 2: PowerShell & Start Menu
    if (level >= 2) {
        // PowerShell Logic (Existing)
        try {
            const psCommand = `
                $apps = Get-ItemProperty "HKLM:\\Software\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\*" -EA SilentlyContinue;
                $found = $apps | Where-Object { $_.DisplayName -like "*${searchName}*" } | Select-Object -First 1;
                if ($found.InstallLocation) { Write-Output $found.InstallLocation }
            `;
            const { stdout } = await new Promise((resolve) => {
                exec(`powershell -NoProfile -Command "${psCommand}"`, (e, out) => resolve({ stdout: out.trim() }));
            });
            if (stdout && fs.existsSync(stdout)) {
                 const exePath = await findMainExecutableAsync(stdout, searchName);
                 if (exePath) return exePath;
            }
        } catch(e) {}
        
        // Start Menu Search
        const startMenuPaths = [
            path.join(process.env.ProgramData || 'C:\\ProgramData', 'Microsoft', 'Windows', 'Start Menu', 'Programs'),
            path.join(app.getPath('home'), 'AppData', 'Roaming', 'Microsoft', 'Windows', 'Start Menu', 'Programs')
        ];

        for (const variant of searchVariants) {
            for (const menuPath of startMenuPaths) {
                if (!fs.existsSync(menuPath)) continue;
                // Simplified Async Check for Start Menu (Depth limited to 2)
                const found = await findAppFolderRecursiveAsync(menuPath, variant, 0, 2); 
                if (found) return found; 
            }
        }
    }

    // LEVEL 3: Common User Paths (Desktop / WindowsApps)
    if (level >= 3) {
        const desktopPaths = [
            path.join(app.getPath('home'), 'Desktop'),
            path.join(process.env.PUBLIC || 'C:\\Users\\Public', 'Desktop')
        ];
        // Check Desktops
        for (const dPath of desktopPaths) {
            try {
                if(fs.existsSync(dPath)) {
                    const files = await fsPromises.readdir(dPath);
                    const lnk = files.find(f => f.toLowerCase().includes(searchName) && f.endsWith('.lnk'));
                    // Note: Resolving LNKs reliably often requires additional tools/powershell, 
                    // but finding the LNK itself is a good indicator.
                }
            } catch(e){}
        }
    }

    // LEVEL 4: High Aggression (Standard Program Files)
    if (level >= 4) {
        const commonPaths = [
            process.env.ProgramFiles,
            process.env["ProgramFiles(x86)"],
            path.join(app.getPath('home'), 'AppData', 'Local', 'Programs')
        ];
        
        for (const variant of searchVariants) {
            for (const basePath of commonPaths) {
                if (!basePath) continue;
                // Limit depth to 2 for High
                const foundFolder = await findAppFolderRecursiveAsync(basePath, variant, 0, 2);
                if (foundFolder) {
                    const exePath = await findMainExecutableAsync(foundFolder, variant);
                    if (exePath) return exePath;
                }
            }
        }
    }

    // LEVEL 5: Extreme (Deep Recursive Search)
    if (level >= 5) {
        const deepSearchPaths = [
            'C:\\ProgramData',
            path.join(app.getPath('home'), 'AppData', 'Local'),
            path.join(app.getPath('home'), 'AppData', 'Roaming'),
        ];
        
        for (const variant of searchVariants) {
            for (const basePath of deepSearchPaths) {
                if (!basePath) continue;
                // Depth 4 for Extreme
                const foundFolder = await findAppFolderRecursiveAsync(basePath, variant, 0, 4);
                if (foundFolder) {
                    const exePath = await findMainExecutableAsync(foundFolder, variant);
                    if (exePath) return exePath;
                }
            }
        }
    }
    
    return null;
});

ipcMain.removeAllListeners("install-app");
ipcMain.on("install-app", (event, filePath) => {
  installAppLogic(filePath);
});

// Load settings on startup
loadAppSettings();

app.whenReady().then(async () => {
    try {
        // Clear all storage types including service workers
        await session.defaultSession.clearStorageData({
            storages: [
                'appcache',
                'cookies',
                'filesystem',
                'indexdb',
                'localstorage',
                'shadercache',
                'websql',
                'serviceworkers',
                'cachestorage'
            ]
        });
        
        // Clear cache
        await session.defaultSession.clearCache();
        
        console.log('✅ All storage and cache cleared successfully');
    } catch (error) {
        // Don't fail silently - log but continue
        console.log('⚠️ Storage cleanup warning (non-critical):', error.message);
    }
    
    createWindow();
    createTray();

    // 🎯 FIX: Attach the download listener to the webview's session (persist:main)
    const webviewSession = session.fromPartition('persist:main');
    setupDownloadHandler(webviewSession);
    
    // OPTIONAL: Keep the defaultSession listener for any non-webview downloads
    setupDownloadHandler(session.defaultSession);
});

app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    if (!appSettings.minimizeToTray) {
      app.quit();
    }
  }
});

app.on('quit', async () => {
    try {
        // Final cleanup
        await session.defaultSession.clearStorageData({
            storages: ['serviceworkers', 'cachestorage']
        });
        console.log('✅ Final cleanup completed');
    } catch (error) {
        // Suppress errors on quit
    }
});
// ✅ NEW: Restart app handler
ipcMain.on("restart-app", () => {
    app.relaunch();
    app.exit(0);
});

// ✅ NEW: Get active renderer
ipcMain.handle("get-active-renderer", () => {
    return appSettings.syncEnabled ? "firebase" : "normal";
});