const { contextBridge, ipcRenderer } = require("electron");

/**
 * FINAL PRELOAD BRIDGE - SWITCHES RENDERERS BASED ON SYNC SETTING
 */

// Instance management
let preloadInstanceId = Math.random().toString(36).substr(2, 9);
let isMainInstance = window === window.top;
let isInitialized = false;
let appMode = 'normal'; // 'normal' or 'firebase'

console.log(`[Preload ${preloadInstanceId}] - Main: ${isMainInstance}, URL: ${window.location.href}`);

// Helper: Safe IPC invoke with timeout
const safeInvoke = async (channel, ...args) => {
    try {
        const timeoutPromise = new Promise((_, reject) =>
            setTimeout(() => reject(new Error(`IPC timeout: ${channel}`)), 10000)
        );
        const result = await Promise.race([
            ipcRenderer.invoke(channel, ...args),
            timeoutPromise
        ]);
        return result;
    } catch (error) {
        console.error(`[Preload ${preloadInstanceId}] IPC invoke error on ${channel}:`, error);
        throw error;
    }
};

// Helper: Safe IPC send
const safeSend = (channel, data) => {
    try {
        ipcRenderer.send(channel, data);
        return true;
    } catch (error) {
        console.error(`[Preload ${preloadInstanceId}] IPC send error on ${channel}:`, error);
        return false;
    }
};

// Helper: Safe listener
const safeListener = (channel, callback) => {
    const wrappedCallback = (event, data) => {
        try {
            callback(data);
        } catch (error) {
            console.error(`[Preload ${preloadInstanceId}] Listener error on ${channel}:`, error);
        }
    };
    
    ipcRenderer.on(channel, wrappedCallback);
    return () => ipcRenderer.removeListener(channel, wrappedCallback);
};

// ✅ NEW: Wait for DOM to be ready
function waitForDOM() {
    return new Promise((resolve) => {
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', resolve);
        } else {
            resolve();
        }
    });
}

// ✅ NEW: Determine which renderer to load
async function determineAndLoadRenderer() {
    try {
        // Get sync setting from main process
        const settings = await safeInvoke("get-app-settings");
        const syncEnabled = settings?.syncEnabled || false;
        
        appMode = syncEnabled ? 'firebase' : 'normal';
        console.log(`[Preload] Sync setting: ${syncEnabled}, Mode: ${appMode}`);
        
        // Set global variable for renderers to check
        window.__APP_MODE = appMode;
        window.__SYNC_ENABLED = syncEnabled;
        
        // Load the appropriate renderer script
        const rendererScript = appMode === 'firebase' ? 'firebase-renderer.js' : 'renderer.js';
        console.log(`[Preload] Loading: ${rendererScript}`);
        
        // Create and inject script
        const script = document.createElement('script');
        script.src = rendererScript;
        script.type = 'text/javascript';
        
        document.head.appendChild(script);
        
        script.onload = () => {
            console.log(`[Preload] ✅ ${rendererScript} loaded successfully`);
        };
        
        script.onerror = (error) => {
            console.error(`[Preload] ❌ Failed to load ${rendererScript}:`, error);
            // Load fallback renderer
            loadFallbackRenderer();
        };
        
    } catch (error) {
        console.error('[Preload] Error determining renderer:', error);
        loadFallbackRenderer();
    }
}

function loadFallbackRenderer() {
    console.log('[Preload] Loading fallback renderer.js');
    const script = document.createElement('script');
    script.src = 'renderer.js';
    document.head.appendChild(script);
    window.__APP_MODE = 'normal';
    window.__SYNC_ENABLED = false;
}

// ✅ CLEAN API Object - NO CHANGES FROM YOUR ORIGINAL
const cleanAPI = {
    // ============================================
    // INSTANCE MANAGEMENT
    // ============================================
    instance: {
        getId: () => preloadInstanceId,
        isMain: () => isMainInstance,
        cleanup: () => {
            console.log(`[Preload ${preloadInstanceId}] Cleaning up instance`);
            isInitialized = false;
        }
    },

    // ============================================
    // DOWNLOAD MANAGEMENT
    // ============================================
    downloads: {
       onStart: (callback) => {
        console.log('[Preload] 📥 Registering download-start listener');
        return safeListener("download-start", (data) => {
            console.log('[Preload] 🎯 download-start received:', data);
            callback(data);
        });
    },
    
    onProgress: (callback) => {
        console.log('[Preload] 📊 Registering download-progress listener');
        return safeListener("download-progress", (data) => {
            console.log('[Preload] 📈 download-progress received:', data);
            callback(data);
        });
    },
    
    onComplete: (callback) => {
        console.log('[Preload] ✅ Registering download-complete listener');
        return safeListener("download-complete", (data) => {
            console.log('[Preload] 🎉 download-complete received:', data);
            callback(data);
        });
    },
    
    onPaused: (callback) => safeListener("download-paused", callback),
    onResumed: (callback) => safeListener("download-resumed", callback),
    
    onFileDeleted: (callback) => safeListener("download-file-deleted", callback),
    
    pause: (filename) => safeSend("pause-download", filename),
    resume: (filename) => safeSend("resume-download", filename),
    cancel: (filename) => safeSend("cancel-download", filename),
        
        getHistory: async () => {
            try {
                const history = await safeInvoke("get-download-history");
                return Array.isArray(history) ? history : [];
            } catch (error) {
                console.error('[Preload] Failed to get download history:', error);
                return [];
            }
        },
        
        saveHistory: (historyData) => {
            if (!Array.isArray(historyData)) {
                console.warn('[Preload] Invalid history data, expected array');
                return false;
            }
            return safeSend("save-download-history", historyData);
        },
        
        clearHistory: () => safeSend("clear-download-history")
    },

    // ============================================
    // APPLICATION MANAGEMENT
    // ============================================
    apps: {
        install: (filePath) => {
            if (typeof filePath !== 'string' || !filePath.trim()) {
                console.error('[Preload] Invalid file path for installation');
                return false;
            }
            return safeSend("install-app", filePath.trim());
        },

        uninstall: (appName) => {
            if (typeof appName !== 'string' || !appName.trim()) {
                console.error('[Preload] Invalid app name for uninstallation');
                return false;
            }
            return safeSend("uninstall-app", appName.trim());
        },

        open: async (appName) => {
            if (typeof appName !== 'string' || !appName.trim()) {
                console.error('[Preload] Invalid app name for opening');
                return false;
            }
            try {
                const result = await safeInvoke("open-app", appName.trim());
                return result === true;
            } catch (error) {
                console.error('[Preload] Failed to open app:', error);
                return false;
            }
        },

        findPath: async (appName, aggression = 5) => {
            if (typeof appName !== 'string' || !appName.trim()) {
                console.error('[Preload] Invalid app name for path search');
                return null;
            }
            try {
                const result = await safeInvoke("find-app-path", appName.trim(), aggression);
                return typeof result === 'string' ? result : null;
            } catch (error) {
                console.error('[Preload] Failed to find app path:', error);
                return null;
            }
        },

        getInfo: async () => {
            try {
                const info = await safeInvoke("get-app-info-data");
                return info || { appName: "Unknown", publisher: "N/A" };
            } catch (error) {
                console.error('[Preload] Failed to get app info:', error);
                return { appName: "Error", publisher: "N/A" };
            }
        },

        onInstallStart: (callback) => safeListener("install-start-signal", callback),
        onShowInstallModal: (callback) => safeListener("show-install-modal", callback)
    },

    // ============================================
    // FILE OPERATIONS
    // ============================================
    files: {
        openLocation: (filePath) => {
            if (typeof filePath !== 'string' || !filePath.trim()) {
                console.error('[Preload] Invalid file path');
                return false;
            }
            return safeSend("open-file-location", filePath.trim());
        },

        delete: (filePath) => {
            if (typeof filePath !== 'string' || !filePath.trim()) {
                console.error('[Preload] Invalid file path for deletion');
                return false;
            }
            return safeSend("delete-file", filePath.trim());
        }
    },

    // ============================================
    // WINDOW CONTROLS
    // ============================================
    window: {
        control: (action) => {
            const validActions = ['minimize', 'maximize', 'close'];
            if (!validActions.includes(action)) {
                console.error(`[Preload] Invalid window action: ${action}`);
                return false;
            }
            return safeSend("control-window", action);
        },

        minimize: () => safeSend("control-window", "minimize"),
        maximize: () => safeSend("control-window", "maximize"),
        close: () => safeSend("control-window", "close"),

        getState: async () => {
            try {
                const state = await safeInvoke("get-initial-window-state");
                return state || { isMaximized: false };
            } catch (error) {
                console.error('[Preload] Failed to get window state:', error);
                return { isMaximized: false };
            }
        },

        onStateChange: (callback) => safeListener("window-state-change", callback)
    },

    // ============================================
    // SETTINGS MANAGEMENT
    // ============================================
    settings: {
        update: (settings) => {
            if (typeof settings !== 'object' || settings === null) {
                console.error('[Preload] Invalid settings object');
                return false;
            }
            return safeSend("update-app-settings", settings);
        },

        get: async () => {
            try {
                const settings = await safeInvoke("get-app-settings");
                // Ensure syncEnabled is included
                return {
                    popupBeforeInstall: true,
                    autoInstall: false,
                    runAsAdmin: false,
                    startupLaunch: false,
                    minimizeToTray: false,
                    ecoMode: false,
                    theme: 'default',
                    syncEnabled: false,
                    ...settings
                };
            } catch (error) {
                console.error('[Preload] Failed to get settings:', error);
                return {
                    popupBeforeInstall: true,
                    autoInstall: false,
                    runAsAdmin: false,
                    startupLaunch: false,
                    minimizeToTray: false,
                    ecoMode: false,
                    theme: 'default',
                    syncEnabled: false
                };
            }
        }
    },

    // ============================================
    // USER PROFILE MANAGEMENT
    // ============================================
    profile: {
        get: async () => {
            try {
                const profile = await safeInvoke("get-user-profile-data");
                return profile;
            } catch (error) {
                console.error('[Preload] Failed to get user profile:', error);
                return null;
            }
        },

        save: (profileData) => {
            if (typeof profileData !== 'object' || !profileData || !profileData.uid) {
                console.error('[Preload] Invalid profile data, uid required');
                return false;
            }
            return safeSend("save-user-profile", profileData);
        },

        clear: async () => {
            try {
                const result = await safeInvoke("clear-user-profile");
                return result || { success: true };
            } catch (error) {
                console.error('[Preload] Failed to clear user profile:', error);
                return { success: false, error: error.message };
            }
        },
        
        getCurrentId: async () => {
            try {
                const uid = await safeInvoke("get-current-user-id");
                return uid || 'guest';
            } catch (error) {
                console.error('[Preload] Failed to get current user ID:', error);
                return 'guest';
            }
        }
    },

    // ============================================
    // FIREBASE CONFIGURATION
    // ============================================
    firebase: {
        getConfig: async () => {
            try {
                const config = await safeInvoke("get-firebase-config");
                return config;
            } catch (error) {
                console.error('[Preload] Failed to get Firebase config:', error);
                return null;
            }
        }
    },

    // ============================================
    // SYSTEM UTILITIES
    // ============================================
    system: {
        toggleDNS: (enable) => safeSend("toggle-dns", Boolean(enable)),

        clearAppData: async () => {
            try {
                const result = await safeInvoke("clear-app-data");
                return result || { success: false, error: 'Unknown error' };
            } catch (error) {
                console.error('[Preload] Failed to clear app data:', error);
                return { success: false, error: error.message };
            }
        },
        
        // ✅ ADD THIS: Restart app function
        restartApp: () => safeSend("restart-app")
    },

    // ============================================
    // FIREBASE MANAGER REFERENCE
    // ============================================
    firebaseManager: {
        getInstance: () => {
            if (window.firebaseManager && isMainInstance) {
                console.log(`[Preload ${preloadInstanceId}] Returning Firebase manager from main instance`);
                return window.firebaseManager;
            }
            console.log(`[Preload ${preloadInstanceId}] No Firebase manager access - not main instance`);
            return null;
        },
        
        isAvailable: () => {
            return !!(window.firebaseManager && isMainInstance);
        },
        
        initialize: async () => {
            if (window.firebaseManager && isMainInstance) {
                return window.firebaseManager;
            }
            return null;
        }
    },

    // ============================================
    // LEGACY COMPATIBILITY
    // ============================================
    onDownloadStart: (fn) => safeListener("download-start", fn),
    onDownloadProgress: (fn) => safeListener("download-progress", fn),
    onDownloadComplete: (fn) => safeListener("download-complete", fn),
    onDownloadPaused: (fn) => safeListener("download-paused", fn),
    onDownloadResumed: (fn) => safeListener("download-resumed", fn),
    
    pauseDownload: (f) => safeSend("pause-download", f),
    resumeDownload: (f) => safeSend("resume-download", f),
    cancelDownload: (f) => safeSend("cancel-download", f),
    
    onShowInstallModal: (fn) => safeListener("show-install-modal", fn),
    onOpenUrlInNewTab: (fn) => safeListener("open-url-in-new-tab", fn),
    onInstallStartSignal: (fn) => safeListener("install-start-signal", fn),
    onWindowStateChange: (fn) => safeListener("window-state-change", fn),

    installApp: (path) => safeSend("install-app", path),
    uninstallApp: (appName) => safeSend("uninstall-app", appName),
    openFileLocation: (filePath) => safeSend("open-file-location", filePath),
    deleteFile: (filePath) => safeSend("delete-file", filePath),
    openApp: (appName) => safeInvoke("open-app", appName),
    
    updateSettings: (settings) => safeSend("update-app-settings", settings),
    toggleDNS: (enable) => safeSend("toggle-dns", enable),
    
    controlWindow: (action) => safeSend("control-window", action),
    findPath: async (appName, aggression = 5) => {
    try {
        const result = await safeInvoke("find-app-path", appName.trim(), aggression);
        return typeof result === 'string' ? result : null;
    } catch (error) {
        console.error('[Preload] Failed to find app path:', error);
        return null;
    }
},
    getInitialWindowState: () => safeInvoke("get-initial-window-state"),
    findAppPath: (appName, aggression) => safeInvoke("find-app-path", appName, aggression),
    getAppInfo: () => safeInvoke("get-app-info-data"),
    getFirebaseConfig: () => safeInvoke("get-firebase-config"),
    getUserProfile: () => safeInvoke("get-user-profile-data"),
    saveUserProfile: (profile) => safeSend("save-user-profile", profile),
    clearUserProfile: () => safeInvoke("clear-user-profile"),
    getDownloadHistory: () => safeInvoke("get-download-history"),
    saveDownloadHistory: (data) => safeSend("save-download-history", data),
    clearDownloadHistory: () => safeSend("clear-download-history"),
    getAppSettings: () => safeInvoke("get-app-settings"),
    
    // ✅ ADD THIS: Restart app legacy function
    restartApp: () => safeSend("restart-app"),

    // ============================================
    // UTILITY FUNCTIONS
    // ============================================
    utils: {
        isElectron: () => true,
        getPlatform: () => process.platform,
        log: (message, data = null) => {
            console.log(`[Renderer -> Main] ${message}`, data);
        },
        
        isFirebaseAvailable: () => {
            return !!(window.firebaseManager && isMainInstance);
        },
        
        getFirebaseManager: () => {
            return (window.firebaseManager && isMainInstance) ? window.firebaseManager : null;
        },
        
        getInstanceInfo: () => {
            return {
                id: preloadInstanceId,
                isMain: isMainInstance,
                url: window.location.href
            };
        },
        
        // ✅ NEW: Get current app mode
        getAppMode: () => appMode,
        isFirebaseMode: () => appMode === 'firebase'
    }
};

// ============================================
// SAFE EXPOSURE LOGIC
// ============================================

const shouldExposeAPI = () => {
    if (isMainInstance) {
        return true;
    }
    
    const isAppStoreDomain = window.location.hostname.includes('www.naroapp.store') ||
                            window.location.hostname.includes('localhost') ||
                            window.location.hostname.includes('127.0.0.1');
    
    if (isAppStoreDomain) {
        console.log(`[Preload ${preloadInstanceId}] Exposing to app store domain: ${window.location.hostname}`);
        return true;
    }
    
    console.log(`[Preload ${preloadInstanceId}] Skipping exposure - external domain: ${window.location.hostname}`);
    return false;
};

// ✅ FIXED: Initialize everything properly
async function initializePreload() {
    try {
        if (shouldExposeAPI() && !isInitialized) {
            // 1. Expose API
            contextBridge.exposeInMainWorld("appAPI", cleanAPI);
            isInitialized = true;
            console.log(`[Preload ${preloadInstanceId}] ✅ API exposed successfully`);
            
            // 2. Wait for DOM
            await waitForDOM();
            console.log('[Preload] DOM ready');
            
            // 3. Load appropriate renderer
            await determineAndLoadRenderer();
            
        } else if (isInitialized) {
            console.log(`[Preload ${preloadInstanceId}] ⚠️ Already initialized`);
        }
    } catch (error) {
        console.error(`[Preload ${preloadInstanceId}] ❌ Initialization error:`, error);
    }
}

// Start initialization
initializePreload();

// Security freezing
if (isInitialized) {
    Object.freeze(cleanAPI);
    Object.freeze(cleanAPI.instance);
    Object.freeze(cleanAPI.downloads);
    Object.freeze(cleanAPI.apps);
    Object.freeze(cleanAPI.files);
    Object.freeze(cleanAPI.window);
    Object.freeze(cleanAPI.settings);
    Object.freeze(cleanAPI.profile);
    Object.freeze(cleanAPI.firebase);
    Object.freeze(cleanAPI.system);
    Object.freeze(cleanAPI.firebaseManager);
    Object.freeze(cleanAPI.utils);
}

console.log(`[Preload ${preloadInstanceId}] ✅ Preload bridge initialized`);