const homeBtn = document.getElementById("homeBtn");
const backBtn = document.getElementById("backBtn");
const forwardBtn = document.getElementById("forwardBtn");
const reloadBtn = document.getElementById("reloadBtn");
const tabBar = document.getElementById("tabBar");
const addTabBtn = document.getElementById("addTabBtn");
const loadingScreen = document.getElementById("loadingScreen");

// Toggles
const togglePopup = document.getElementById("togglePopup");
const toggleAuto = document.getElementById("toggleAuto");
const toggleDNS = document.getElementById("toggleDNS");

// Download UI
const downloadBtn = document.getElementById("downloadBtn");
const downloadPopover = document.getElementById("downloadPopover");
const dlBadge = document.getElementById("dlBadge");

// Profile UI
const profileBtn = document.getElementById("profileBtn");
const profileDropdown = document.getElementById("profileDropdown");
const profileImg = document.getElementById("profileImg");
const profilePlaceholder = document.getElementById("profilePlaceholder");
const profileName = document.getElementById("profileName");
const profileEmail = document.getElementById("profileEmail");

// Dropdown Profile Elements
const dropdownProfileImg = document.getElementById("dropdownProfileImg");
const dropdownProfileIcon = document.getElementById("dropdownProfileIcon");

// Modals
const installModal = document.getElementById("installModal");
const laterBtn = document.getElementById("laterBtn");
const modalFileName = document.getElementById("modalFileName");
const installNowBtn = document.getElementById("installNowBtn");
const closeInstallModalBtn = document.getElementById("closeInstallModalBtn");

const appInfoTabBtn = document.getElementById("appInfoTabBtn");
const speedometerContainer = document.createElement('div');
speedometerContainer.id = 'speedometer';
speedometerContainer.innerHTML = `
    <div class="speed-display">
        <div class="speed-value" id="speedValue">0</div>
        <div class="speed-unit">MB/s</div>
    </div>
    <div class="signal-bars">
        <div class="signal-bar"></div>
        <div class="signal-bar"></div>
        <div class="signal-bar"></div>
        <div class="signal-bar"></div>
    </div>
`;

const debugLog = {
    app: (message, data = null) => {
        console.log(`📱 [App] ${message}`, data || '');
    },
    auth: (message, data = null) => {
        console.log(`🔐 [Auth] ${message}`, data || '');
    }
};

const toolbarRight = document.getElementById('toolbarRight');
toolbarRight.insertBefore(speedometerContainer, appInfoTabBtn);

// Network speed monitoring
let lastBytes = 0;
let lastTime = Date.now();

function updateNetworkSpeed() {
    if (navigator.connection) {
        const connection = navigator.connection;
        const downlink = connection.downlink || 0;
        const speedMBps = (downlink / 8).toFixed(2);

        document.getElementById('speedValue').innerText = speedMBps;

        const bars = document.querySelectorAll('.signal-bar');
        const signalStrength = Math.min(Math.floor(downlink / 5), 4);

        bars.forEach((bar, index) => {
            if (index < signalStrength) {
                bar.classList.add('active');
            } else {
                bar.classList.remove('active');
            }
        });
    }
}

setInterval(updateNetworkSpeed, 2000);
updateNetworkSpeed();

const appInfoModal = document.getElementById("appInfoModal");
const downloadDetailModal = document.getElementById("downloadDetailModal");
const detailAppName = document.getElementById("detailAppName");
const detailSize = document.getElementById("detailSize");
const detailDownloadedDate = document.getElementById("detailDownloadedDate");
const detailLocationPath = document.getElementById("detailLocationPath");
const detailInstalledStatus = document.getElementById("detailInstalledStatus");
const detailRunInstallerBtn = document.getElementById("detailRunInstallerBtn");
const detailOpenFolderBtn = document.getElementById("detailOpenFolderBtn");
const detailDeleteFileBtn = document.getElementById("detailDeleteFileBtn");
const detailCopyBtn = document.getElementById('detailCopyBtn');

// Account Settings Modal
const accountSettingsModal = document.getElementById("accountSettingsModal");
const closeAccountSettings = document.getElementById("closeAccountSettings");
const profileAccountSettingsBtn = document.getElementById("profileAccountSettingsBtn");

// Appearance Modal
const appearanceModal = document.getElementById("appearanceModal");
const closeAppearance = document.getElementById("closeAppearance");
const profileAppearanceBtn = document.getElementById("profileAppearanceBtn");

// Sync Settings Elements
const profileSyncBtn = document.getElementById("profileSyncBtn");
const syncSettingsModal = document.getElementById("syncSettingsModal");
const closeSyncSettings = document.getElementById("closeSyncSettings");
const settingSyncToggle = document.getElementById("settingSyncToggle");
const aggressionSlider = document.getElementById("aggressionSlider");
const aggressionDisplay = document.getElementById("aggressionDisplay");

// Sign Out Button
const profileSignOutBtn = document.getElementById("profileSignOutBtn");

// Custom Confirm Dialog
const customConfirm = document.getElementById("customConfirm");
const confirmTitle = document.getElementById("confirmTitle");
const confirmMessage = document.getElementById("confirmMessage");
const confirmIcon = document.getElementById("confirmIcon");
const confirmYes = document.getElementById("confirmYes");
const confirmNo = document.getElementById("confirmNo");

document.getElementById("addTabBtn").addEventListener('click', () => addNewTab(defaultLoadUrl));

let minimizeBtn, maximizeBtn, closeBtn;
const defaultLoadUrl = "https://www.naroapp.store/";

let tabs = [];
let activeWebviewId = null;
let tabCounter = 0;
let lastDownloadedFile = null;
let isMaximized = false;
let currentUser = null;
let isSigningOut = false;
let downloadHistory = [];

// --- SYNC CONFIGURATION (Local Only) ---
let syncConfig = {
    enabled: localStorage.getItem('syncEnabled') === 'true',
    aggression: parseInt(localStorage.getItem('syncAggression') || '5')
};

const aggressionLabels = {
    1: "Low (Registry Only - Fast)",
    2: "Normal (Registry + Start Menu)",
    3: "Medium (Reg + Start + Desktop)",
    4: "High (Deep Search in Common Folders)",
    5: "Extreme (Full Recursive Search)"
};

// Initialize Sync UI
if (settingSyncToggle) settingSyncToggle.checked = syncConfig.enabled;
if (aggressionSlider) {
    aggressionSlider.value = syncConfig.aggression;
    aggressionDisplay.innerText = "Current: " + aggressionLabels[syncConfig.aggression];
}

// --- HELPER: ROBUST KEY GENERATION ---
function getCleanAppName(filename) {
    if (!filename) return "App";
    let name = filename.replace(/\.(exe|msi|zip|rar|iso)$/i, '');
    let parts = name.split(/_| v\d/);
    let cleanName = parts[0] ? parts[0] : name;
    return cleanName.trim();
}

// --- HELPER: MATCH CLEANER ---
function getCleanMatchName(rawName) {
    if (!rawName) return "";
    return rawName.toLowerCase()
        .replace(/ pc$/i, '')
        .replace(/ edition$/i, '')
        .replace(/ app$/i, '')
        .replace(/ software$/i, '')
        .replace(/[^a-z0-9]/g, '');
}

// --- SAVE DOWNLOAD HISTORY TO DISK ---
function saveDownloadHistoryToDisk() {
    try {
        if (window.appAPI && window.appAPI.saveDownloadHistory) {
            window.appAPI.saveDownloadHistory(downloadHistory);
            console.log('💾 Download history saved:', downloadHistory.length, 'items');
        }
    } catch (e) {
        console.error("Error saving download history:", e);
    }
}
function throttle(func, delay) {
    let lastCall = 0;
    return function(...args) {
        const now = Date.now();
        if (now - lastCall >= delay) {
            lastCall = now;
            return func.apply(this, args);
        }
    };
}
// --- RENDER SIDEBAR HISTORY (FIXED VERSION) ---
function renderSidebarHistory() {
    const sidebarContent = document.getElementById('sidebarContent');
    const noDownloads = document.getElementById('noDownloads');
    
    // Check if there are any downloads
    if (!downloadHistory || downloadHistory.length === 0) {
        if (noDownloads) noDownloads.style.display = 'flex';
        if (sidebarContent) {
            const existingItems = sidebarContent.querySelectorAll('.sidebar-download-item');
            existingItems.forEach(item => item.remove());
        }
        return;
    }
    
    if (noDownloads) noDownloads.style.display = 'none';
    
    // Clear existing sidebar items to re-render
    const existingItems = sidebarContent ? sidebarContent.querySelectorAll('.sidebar-download-item') : [];
    existingItems.forEach(item => item.remove());
    
    if (!sidebarContent) {
        console.error('❌ sidebarContent not found!');
        return;
    }
    
    // Render each history item
    downloadHistory.forEach((item, index) => {
        const dlItem = document.createElement('div');
        dlItem.className = 'sidebar-download-item';
        dlItem.dataset.index = index;
        dlItem.dataset.filename = item.filename;
        dlItem.dataset.path = item.path || 'N/A';
        dlItem.dataset.size = item.size || 'N/A';
        dlItem.dataset.date = item.date || 'Unknown';
        
        dlItem.style.pointerEvents = 'auto';
        dlItem.style.cursor = 'pointer';
        dlItem.style.position = 'relative';
        dlItem.style.zIndex = '10';
        
        const isComplete = item.progress === 100;
        const isFailed = item.progress === -1;
        const isActive = !isComplete && !isFailed;
        const isPaused = item.isPaused === true;
        
        const statusText = isComplete ? 'Completed' : 
                          isFailed ? 'Failed' : 
                          isPaused ? 'Paused' : `${item.progress}%`;
        
        const statusColor = isComplete ? '#00e676' : 
                           isFailed ? '#ff4444' : 
                           isPaused ? '#ffa500' : '#00d4ff';

        // Control buttons based on state
        let controlsHtml = '';
        if (isActive) {
            // Active Download: Pause/Resume and Cancel
            const pauseIcon = isPaused ? 'fa-play' : 'fa-pause';
            const pauseTitle = isPaused ? 'Resume' : 'Pause';
            controlsHtml = `
                <button class="action-btn pause-btn" title="${pauseTitle}">
                    <i class="fa-solid ${pauseIcon}"></i>
                </button>
                <button class="action-btn cancel-btn" title="Cancel">
                    <i class="fa-solid fa-xmark"></i>
                </button>
            `;
        } else {
            // Completed or Failed: Delete button
            controlsHtml = `
                <button class="action-btn delete-btn" title="Remove from list">
                    <i class="fa-solid fa-trash"></i>
                </button>
            `;
        }
        
        dlItem.innerHTML = `
            <div class="sidebar-dl-content">
                <div class="sidebar-dl-name">${item.filename}</div>
                <div class="sidebar-dl-info">
                    <span>${item.size || 'N/A'}</span>
                    <span class="sidebar-dl-status" style="color: ${statusColor}">${statusText}</span>
                </div>
            </div>
            <div class="sidebar-dl-actions">
                ${controlsHtml}
            </div>
        `;
        
        // --- CLICK HANDLERS ---
        
        // 1. Pause / Resume
        const pauseBtn = dlItem.querySelector('.pause-btn');
        if (pauseBtn) {
            pauseBtn.addEventListener('click', (e) => {
                e.stopPropagation(); // Prevent opening modal
                const filename = item.filename;
                if (isPaused) {
                    window.appAPI.resumeDownload(filename);
                } else {
                    window.appAPI.pauseDownload(filename);
                }
            });
        }
        
        // 2. Cancel - CUSTOM CONFIRM
        const cancelBtn = dlItem.querySelector('.cancel-btn');
        if (cancelBtn) {
            cancelBtn.addEventListener('click', async (e) => {
                e.stopPropagation();
                const confirmed = await showConfirmDialog(
                    "Cancel Download",
                    `Are you sure you want to cancel the download for "${item.filename}"?`,
                    "🛑"
                );
                
                if (confirmed) {
                    window.appAPI.cancelDownload(item.filename);
                    // Update UI immediately to failed/cancelled state handled by download-complete event
                }
            });
        }
        
        // 3. Delete (History) - CUSTOM CONFIRM
        const deleteBtn = dlItem.querySelector('.delete-btn');
        if (deleteBtn) {
            deleteBtn.addEventListener('click', async (e) => {
                e.stopPropagation();
                const confirmed = await showConfirmDialog(
                    "Remove from History",
                    `Are you sure you want to remove "${item.filename}" from your history?`,
                    "🗑️"
                );
                
                if (confirmed) {
                    downloadHistory.splice(index, 1);
                    saveDownloadHistoryToDisk();
                    renderSidebarHistory();
                    updateProfileStats();
                }
            });
        }

        // 4. Main Item Click (Details)
        dlItem.addEventListener('click', function(e) {
            // Only trigger if we didn't click an action button
            if (!e.target.closest('.action-btn')) {
                showDownloadDetailModalFromSidebar(this);
            }
        });
        
        // Hover effects
        dlItem.addEventListener('mouseenter', function() {
            this.style.background = 'rgba(159, 0, 255, 0.15)';
        });
        dlItem.addEventListener('mouseleave', function() {
            this.style.background = '';
        });
        
        sidebarContent.appendChild(dlItem);
    });
}

// ✅ NEW: Show detail modal from sidebar item
async function showDownloadDetailModalFromSidebar(sidebarItem) {
    try {
        console.log('🔍 Opening detail modal for sidebar item');
        
        const fileName = sidebarItem.dataset.filename;
        const downloadPath = sidebarItem.dataset.path;
        const downloadDate = sidebarItem.dataset.date;
        const fileSize = sidebarItem.dataset.size;
        
        console.log('📦 Item data:', { fileName, downloadPath, downloadDate, fileSize });
        
        if (!fileName) {
            console.error('❌ No filename found in dataset');
            return;
        }
        
        detailAppName.innerText = fileName;
        detailDownloadedDate.innerText = downloadDate;
        detailSize.innerText = fileSize;

        const guessedAppName = getCleanAppName(fileName);

        detailInstalledStatus.innerText = 'Checking...';
        detailInstalledStatus.style.color = '#ffc107';

        let installedPath = null;
        if (downloadPath !== 'N/A') {
            installedPath = await window.appAPI.findPath(guessedAppName, syncConfig.aggression);
            console.log(`🔍 App search: "${guessedAppName}" | Found: ${installedPath || 'Not found'}`);
        }

        let isInstalled = !!installedPath;
        let currentPath = isInstalled ? installedPath : downloadPath;

        if (isInstalled) {
            detailInstalledStatus.innerText = 'Installed on Device';
            detailInstalledStatus.style.color = '#00e676';
            detailLocationPath.innerText = installedPath;
            detailRunInstallerBtn.innerText = 'Run App';
            detailDeleteFileBtn.innerHTML = '<i class="fa-solid fa-trash"></i> Uninstall';
            detailDeleteFileBtn.classList.remove('btn-danger');
            detailDeleteFileBtn.classList.add('btn-secondary');
            detailDeleteFileBtn.dataset.action = 'uninstall';
            detailOpenFolderBtn.innerHTML = '<i class="fa-solid fa-folder-open"></i> Open Installed Folder';
        } else {
            detailInstalledStatus.innerText = 'Not Installed';
            detailInstalledStatus.style.color = '#d32f2f';
            detailLocationPath.innerText = downloadPath;
            detailRunInstallerBtn.innerText = 'Run Installer';
            detailDeleteFileBtn.innerHTML = '<i class="fa-solid fa-trash"></i> Delete File';
            detailDeleteFileBtn.classList.remove('btn-secondary');
            detailDeleteFileBtn.classList.add('btn-danger');
            detailDeleteFileBtn.dataset.action = 'delete';
            detailOpenFolderBtn.innerHTML = '<i class="fa-solid fa-folder-open"></i> Open Download Folder';
        }

        const isDisabled = downloadPath === 'N/A';
        detailRunInstallerBtn.disabled = isDisabled;
        detailOpenFolderBtn.disabled = isDisabled;
        detailDeleteFileBtn.disabled = isDisabled;
        detailCopyBtn.disabled = isDisabled;

        // Button handlers remain the same...
        detailRunInstallerBtn.onclick = () => {
            if (currentPath && currentPath !== 'N/A') {
                window.appAPI.installApp(currentPath);
                if (!isInstalled) {
                    updateAppInstallationStatus({
                        appName: guessedAppName,
                        filename: fileName,
                        size: fileSize,
                        date: downloadDate,
                        path: currentPath,
                        status: 'installed'
                    });
                    const webview = getActiveWebview();
                    if (webview) {
                        webview.send("install-start-signal", {
                            type: "INSTALL_INITIATED",
                            appName: guessedAppName
                        });
                    }
                }
                downloadDetailModal.style.display = 'none';
            }
        };

        detailOpenFolderBtn.onclick = () => {
            if (currentPath && currentPath !== 'N/A') {
                window.appAPI.openFileLocation(currentPath);
                downloadDetailModal.style.display = 'none';
            }
        };

        detailDeleteFileBtn.onclick = async () => {
            if (detailDeleteFileBtn.dataset.action === 'uninstall') {
                const confirmed = await showConfirmDialog(
                    "Uninstall App",
                    `Are you sure you want to uninstall "${guessedAppName}"?`,
                    "⚠️"
                );
                
                if (confirmed) {
                    window.appAPI.uninstallApp(guessedAppName);
                    updateAppInstallationStatus({
                        appName: guessedAppName,
                        filename: fileName,
                        size: fileSize,
                        date: downloadDate,
                        path: installedPath || downloadPath,
                        status: 'uninstalled'
                    });
                    downloadDetailModal.style.display = 'none';
                }
            } else if (downloadPath && downloadPath !== 'N/A') {
                const confirmed = await showConfirmDialog(
                    "Delete File",
                    `Are you sure you want to delete "${fileName}"?`,
                    "🗑️"
                );
                
                if (confirmed) {
                    window.appAPI.deleteFile(downloadPath);
                    const historyIndex = downloadHistory.findIndex(h => h.filename === fileName);
                    if (historyIndex >= 0) {
                        downloadHistory.splice(historyIndex, 1);
                        saveDownloadHistoryToDisk();
                    }
                    const itemToRemove = document.getElementById(`dl-${fileName}`);
                    if (itemToRemove) itemToRemove.remove();
                    renderSidebarHistory();
                    downloadDetailModal.style.display = 'none';
                }
            }
        };

        detailCopyBtn.onclick = () => {
            const textToCopy = detailLocationPath.innerText;
            navigator.clipboard.writeText(textToCopy).then(() => {
                const originalHtml = '<i class="fa-regular fa-copy"></i> Copy';
                detailCopyBtn.innerHTML = '<i class="fa-solid fa-check"></i> Copied!';
                detailCopyBtn.style.background = '#00e676';
                detailCopyBtn.style.color = '#000';
                setTimeout(() => {
                    detailCopyBtn.innerHTML = originalHtml;
                    detailCopyBtn.style.background = '';
                    detailCopyBtn.style.color = '';
                }, 2000);
            }).catch(err => console.error("Copy failed", err));
        };

        // Show the modal
        console.log('✅ Opening detail modal');
        downloadDetailModal.style.display = 'flex';

    } catch (err) {
        console.error("❌ Error opening detail modal from sidebar:", err);
    }
}
// Sidebar toggle functionality
const sidebarToggle = document.getElementById('sidebarToggle');
const downloadSidebar = document.getElementById('downloadSidebar');
const webBox = document.getElementById('webBox');

if (sidebarToggle && downloadSidebar) {
    sidebarToggle.addEventListener('click', () => {
        if (downloadSidebar.classList.contains('visible')) {
            downloadSidebar.classList.remove('visible');
            downloadSidebar.classList.add('collapsed');
            if (webBox) webBox.classList.add('fullwidth');
        } else {
            downloadSidebar.classList.remove('collapsed');
            downloadSidebar.classList.add('visible');
            if (webBox) webBox.classList.remove('fullwidth');
        }
        
        const icon = sidebarToggle.querySelector('i');
        if (downloadSidebar.classList.contains('collapsed')) {
            icon.className = 'fa-solid fa-chevron-left';
        } else {
            icon.className = 'fa-solid fa-chevron-right';
        }
    });
}

// --- LOAD DOWNLOAD HISTORY FROM DISK ---
// --- LOAD DOWNLOAD HISTORY FROM DISK (FIXED) ---
async function loadDownloadHistoryFromDisk() {
    try {
        const history = await window.appAPI.getDownloadHistory();
        
        if (history && Array.isArray(history)) {
            downloadHistory = history;
            
            // Set download badge visibility
            if (history.length > 0) {
                const activeOrCompleted = history.filter(item => 
                    item.progress === 100 || item.progress > 0
                ).length;
                
                if (activeOrCompleted > 0) {
                    dlBadge.style.display = "block";
                } else {
                    dlBadge.style.display = "none";
                }
            }
            
            // ✅ ONLY render sidebar (no dl-items in popover)
            renderSidebarHistory();
            console.log(`✅ Loaded download history: ${downloadHistory.length} items. Sidebar rendered.`);

        } else {
            downloadHistory = [];
            renderSidebarHistory();
            console.log('⚠️ Download history file empty or corrupt. Initialized history array.');
        }

    } catch (e) {
        console.error("❌ Error loading download history:", e);
        downloadHistory = [];
        renderSidebarHistory();
    }
}
// --- LOAD APP SETTINGS FROM DISK ---
async function loadAppSettingsFromDisk() {
    try {
        const settings = await window.appAPI.getAppSettings();
        if (settings) {
            togglePopup.checked = settings.popupBeforeInstall !== false;
            toggleAuto.checked = settings.autoInstall === true;

            // Sync the toggle with saved settings
            if (settingSyncToggle) {
                const savedSyncState = settings.syncEnabled || localStorage.getItem('syncEnabled') === 'true';
                settingSyncToggle.checked = savedSyncState;
                syncConfig.enabled = savedSyncState;
            }
            
            // Sync aggression slider
            if (aggressionSlider) {
                const savedAggression = parseInt(localStorage.getItem('syncAggression') || '5');
                aggressionSlider.value = savedAggression;
                syncConfig.aggression = savedAggression;
                if (aggressionDisplay) {
                    aggressionDisplay.innerText = "Current: " + aggressionLabels[savedAggression];
                }
            }

            const settingRunAsAdmin = document.getElementById("settingRunAsAdmin");
            const settingStartupLaunch = document.getElementById("settingStartupLaunch");
            const settingMinimizeToTray = document.getElementById("settingMinimizeToTray");
            const settingEcoMode = document.getElementById("settingEcoMode");

            if (settingRunAsAdmin) settingRunAsAdmin.checked = settings.runAsAdmin === true;
            if (settingStartupLaunch) settingStartupLaunch.checked = settings.startupLaunch === true;
            if (settingMinimizeToTray) settingMinimizeToTray.checked = settings.minimizeToTray === true;
            if (settingEcoMode) settingEcoMode.checked = settings.ecoMode === true;

            if (settings.theme) {
                applyTheme(settings.theme);
            }

            if (settings.ecoMode) {
                document.documentElement.setAttribute('data-eco', 'true');
            } else {
                document.documentElement.removeAttribute('data-eco');
            }
        }
    } catch (e) {
        console.error("Error loading app settings:", e);
    }
}

// --- APPLY THEME ---
function applyTheme(themeName) {
    document.documentElement.setAttribute('data-theme', themeName);

    document.querySelectorAll('.theme-option').forEach(option => {
        option.classList.remove('active');
        if (option.dataset.theme === themeName) {
            option.classList.add('active');
        }
    });
}

// --- USER PROFILE LOGIC ---
async function fetchUserProfile() {
    try {
        debugLog.auth('Fetching local user profile...');
        const localProfile = await window.appAPI.getUserProfile();

        if (localProfile && localProfile.uid && localProfile.uid !== 'guest') {
            debugLog.auth('Valid user profile found locally:', localProfile.email);
            currentUser = localProfile;
            updateProfileUI(localProfile);
            return;
        }

        resetProfileUI();

    } catch (e) {
        console.error("Error fetching user profile:", e);
        resetProfileUI();
    }
}

// --- UPDATE PROFILE UI ---
function updateProfileUI(user) {
    debugLog.auth('Updating profile UI for:', user.email);

    profileName.innerText = user.displayName || "User";
    profileEmail.innerText = "Logged in";

    if (user.photoURL) {
        profileImg.src = user.photoURL;
        profileImg.style.display = "block";
        profilePlaceholder.style.display = "none";

        dropdownProfileImg.src = user.photoURL;
        dropdownProfileImg.style.display = "block";
        dropdownProfileIcon.style.display = "none";

        profileImg.onerror = () => {
            profileImg.style.display = "none";
            profilePlaceholder.style.display = "block";
        };

        dropdownProfileImg.onerror = () => {
            dropdownProfileImg.style.display = "none";
            dropdownProfileIcon.style.display = "flex";
        };
    } else {
        profileImg.style.display = "none";
        profilePlaceholder.style.display = "block";
        dropdownProfileImg.style.display = "none";
        dropdownProfileIcon.style.display = "flex";
    }
}

// --- RESET PROFILE UI ---
function resetProfileUI() {
    currentUser = null;
    profileName.innerText = "Guest User";
    profileEmail.innerText = "Not logged in";

    profileImg.style.display = "none";
    profilePlaceholder.style.display = "block";

    dropdownProfileImg.style.display = "none";
    dropdownProfileIcon.style.display = "flex";
}

// --- SYNC FUNCTIONS (LOCAL STUBS) ---
function addDownloadToHistory(filename, path, size) {
    // 1. Ensure the array is saved to disk (IPC call)
    saveDownloadHistoryToDisk(); 

    // 2. Ensure the sidebar is rendered to show the new persistent item
    renderSidebarHistory();
    
    console.log("✅ Download history persistence and rendering ensured.");
}

function updateAppInstallationStatus(appData) {
    // This file is the "offline/standard" renderer.
    // The Firebase update logic has been removed.
    console.log("Updating local app status (No Sync)");
}

// --- UTILITIES ---
function getActiveWebview() {
    return tabs.find(t => t.id === activeWebviewId)?.webview;
}

function showLoadingScreen() {
    loadingScreen.style.opacity = 1;
    loadingScreen.style.pointerEvents = 'auto';
}

function hideLoadingScreen() {
    loadingScreen.style.opacity = 0;
    loadingScreen.style.pointerEvents = 'none';
}

function updateMaximizeIcon() {
    const icon = maximizeBtn.querySelector('i');
    if (!icon) return;
    if (isMaximized) {
        icon.className = 'fa-regular fa-window-restore';
        maximizeBtn.title = 'Restore Down';
    } else {
        icon.className = 'fa-regular fa-window-maximize';
        maximizeBtn.title = 'Maximize';
    }
}

// --- CUSTOM CONFIRM DIALOG ---
function showConfirmDialog(title, message, icon = '⚠️') {
    return new Promise((resolve) => {
        confirmTitle.innerText = title;
        confirmMessage.innerText = message;
        confirmIcon.innerText = icon;
        customConfirm.style.display = 'flex';

        const handleYes = () => {
            customConfirm.style.display = 'none';
            confirmYes.removeEventListener('click', handleYes);
            confirmNo.removeEventListener('click', handleNo);
            resolve(true);
        };

        const handleNo = () => {
            customConfirm.style.display = 'none';
            confirmYes.removeEventListener('click', handleYes);
            confirmNo.removeEventListener('click', handleNo);
            resolve(false);
        };

        confirmYes.addEventListener('click', handleYes);
        confirmNo.addEventListener('click', handleNo);
    });
}

// --- TAB MANAGEMENT ---
function updateNavigationButtons() {
    const activeView = getActiveWebview();
    if (!activeView || typeof activeView.canGoBack !== 'function' || typeof activeView.canGoForward !== 'function') {
        backBtn.disabled = true;
        forwardBtn.disabled = true;
        return;
    }
    try {
        backBtn.disabled = !activeView.canGoBack();
        forwardBtn.disabled = !activeView.canGoForward();
    } catch (e) {
        backBtn.disabled = true;
        forwardBtn.disabled = true;
    }
}

function activateTab(tabId) {
    activeWebviewId = tabId;

    tabs.forEach(t => {
        const isActive = t.id === tabId;
        t.webview.style.opacity = isActive ? '1' : '0';
        t.webview.style.zIndex = isActive ? '1' : '0';

        if (isActive) {
            t.tabBtn.classList.add('active');
        } else {
            t.tabBtn.classList.remove('active');
        }
    });

    updateNavigationButtons();

    const activeWebview = getActiveWebview();
    if (activeWebview) {
        try {
            if (activeWebview.isLoading && typeof activeWebview.isLoading === 'function' && activeWebview.isLoading()) {
                showLoadingScreen();
            } else {
                hideLoadingScreen();
            }
        } catch (e) {
            hideLoadingScreen();
        }
    }
}

function closeTab(tabId) {
    if (tabs.length === 1) {
        window.appAPI.controlWindow('close');
        return;
    }
    const indexToClose = tabs.findIndex(t => t.id === tabId);
    if (indexToClose === -1) return;

    const tabToClose = tabs[indexToClose];
    tabToClose.webview.remove();
    tabToClose.tabBtn.remove();
    tabs.splice(indexToClose, 1);

    if (tabId === activeWebviewId) {
        const newActiveIndex = Math.min(indexToClose, tabs.length - 1);
        activateTab(tabs[newActiveIndex].id);
    }
}

function addNewTab(url = defaultLoadUrl) {
    tabCounter++;
    const newId = `tab-${tabCounter}`;

    const newWebview = document.createElement('webview');
    newWebview.id = `view-${newId}`;

    console.log('🌐 Creating webview with URL:', url);

    newWebview.setAttribute('allowpopups', 'true');
    newWebview.setAttribute('partition', 'persist:main');

    newWebview.src = url;

    webBox.appendChild(newWebview);

    const tabBtn = document.createElement('div');
    tabBtn.className = 'tab';
    tabBtn.innerHTML = `
        <img class="tab-favicon" src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3E%3Cpath fill='%23999' d='M8 0a8 8 0 100 16A8 8 0 008 0z'/%3E%3C/svg%3E" alt="">
        <span class="tab-title">Loading...</span>
        <button class="tab-close" title="Close Tab">&times;</button>
    `;
    tabBar.appendChild(tabBtn);

    const newTab = {
        id: newId,
        webview: newWebview,
        tabBtn: tabBtn
    };
    tabs.push(newTab);

    tabBtn.onclick = () => activateTab(newId);
    tabBtn.querySelector('.tab-close').onclick = (e) => {
        e.stopPropagation();
        closeTab(newId);
    };

    newWebview.addEventListener('new-window', (e) => {
        e.preventDefault();

        const url = e.url;
        console.log('🔗 New window requested:', url);

        const downloadExtensions = ['.exe', '.msi', '.zip', '.rar', '.7z', '.dmg', '.pkg', '.deb', '.rpm', '.apk', '.iso'];
        const isDownload = downloadExtensions.some(ext => url.toLowerCase().endsWith(ext));

        if (isDownload) {
            console.log('📥 Download detected, not opening tab:', url);
            return;
        }

        if (url.startsWith('blob:') || url.startsWith('data:')) {
            console.log('🚫 Ignoring blob/data URL:', url);
            return;
        }

        console.log('✅ Opening in new tab:', url);
        addNewTab(url);
    });

    let blankTabTimeout = setTimeout(() => {
        try {
            const currentUrl = newWebview.getURL();
            const tabTitle = tabBtn.querySelector('.tab-title').innerText;

            if ((!currentUrl || currentUrl === '' || currentUrl === 'about:blank') &&
                (tabTitle === 'Loading...' || tabTitle === '')) {
                console.log('🗑️ Auto-closing blank tab:', newId);
                closeTab(newId);
            }
        } catch (e) {
            console.log('Tab check error:', e);
        }
    }, 3000);

    newWebview.addEventListener('did-finish-load', () => {
        clearTimeout(blankTabTimeout);
    });

    newWebview.addEventListener('destroyed', () => {
        clearTimeout(blankTabTimeout);
    });

    const injectLinkHandler = () => {
        const script = `
            (function() {
                if (window.__linkHandlerInstalled) return;
                window.__linkHandlerInstalled = true;
                
                console.log('✅ Advanced link handler injected');
                
                document.addEventListener('click', function(e) {
                    let target = e.target;
                    
                    while (target && target.tagName !== 'A') {
                        target = target.parentElement;
                    }
                    
                    if (target && target.tagName === 'A') {
                        const href = target.getAttribute('href');
                        const targetAttr = target.getAttribute('target');
                        const currentDomain = window.location.hostname;
                        
                        let isExternal = false;
                        try {
                            if (href && (href.startsWith('http://') || href.startsWith('https://'))) {
                                const linkUrl = new URL(href);
                                isExternal = linkUrl.hostname !== currentDomain;
                            }
                        } catch (err) {}
                        
                        const shouldOpenNewTab = 
                            targetAttr === '_blank' ||
                            e.ctrlKey ||
                            e.metaKey ||
                            e.button === 1 ||
                            isExternal;
                        
                        if (shouldOpenNewTab && href) {
                            e.preventDefault();
                            e.stopPropagation();
                            window.open(href, '_blank');
                            return false;
                        }
                    }
                }, true);
                
                const originalOpen = window.open;
                window.open = function(url, target, features) {
                    return originalOpen.call(window, url, target || '_blank', features);
                };
            })();
        `;

        try {
            newWebview.executeJavaScript(script)
                .then(() => console.log('✅ Link handler injected successfully'))
                .catch(err => console.warn('⚠️ Failed to inject link handler:', err));
        } catch (err) {
            console.warn('⚠️ Could not inject link handler:', err);
        }
    };

    newWebview.addEventListener('did-start-loading', () => {
        if (newId === activeWebviewId) {
            showLoadingScreen();
        }
    });

    newWebview.addEventListener('dom-ready', () => {
        if (newId === activeWebviewId) {
            hideLoadingScreen();
        }
        injectLinkHandler();
    });

    newWebview.addEventListener('did-finish-load', () => {
        injectLinkHandler();
        setTimeout(() => injectLinkHandler(), 1000);
    });

    newWebview.addEventListener('did-fail-load', (event) => {
        console.error('❌ Failed to load:', event);
        if (newId === activeWebviewId) {
            hideLoadingScreen();
        }
        tabBtn.querySelector('.tab-title').innerText = 'Failed to Load';
    });

    newWebview.addEventListener('did-navigate', () => {
        updateNavigationButtons();
        injectLinkHandler();
    });

    newWebview.addEventListener('did-navigate-in-page', () => {
        updateNavigationButtons();
        injectLinkHandler();
    });

    newWebview.addEventListener('page-title-updated', (e) => {
        tabBtn.querySelector('.tab-title').innerText = e.title || 'Untitled';
    });

    newWebview.addEventListener('page-favicon-updated', (e) => {
        const favicon = tabBtn.querySelector('.tab-favicon');
        if (e.favicons && e.favicons.length > 0) {
            favicon.src = e.favicons[0];
            favicon.onerror = () => {
                favicon.src = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3E%3Cpath fill='%23999' d='M8 0a8 8 0 100 16A8 8 0 008 0z'/%3E%3C/svg%3E";
            };
        }
    });

    const contentObserver = setInterval(() => {
        if (newWebview.isLoading && typeof newWebview.isLoading === 'function') {
            if (!newWebview.isLoading()) {
                injectLinkHandler();
            }
        }
    }, 2000);

    newWebview.addEventListener('destroyed', () => {
        clearInterval(contentObserver);
    });

    // ✅ AUTH STATUS CHECK (Stripped of Firebase Sign-In)
    // This only checks if the webview has a user to update the UI header
    const checkAuthStatus = async () => {
        if (isSigningOut) return;

        try {
            const currentUrl = newWebview.getURL();
            const isAppDomain = currentUrl.includes("www.naroapp.store") ||
                currentUrl.includes("firebaseapp.com") ||
                currentUrl.includes("localhost") ||
                currentUrl.includes("127.0.0.1");

            if (!isAppDomain) return;

            const getUserFromPage = async () => {
                try {
                    const code = `
                    (() => {
                        try {
                            const key = Object.keys(localStorage).find(k => k.includes('firebase:authUser'));
                            if (key) {
                                const val = localStorage.getItem(key);
                                if(val) {
                                    const user = JSON.parse(val);
                                    if (user && user.uid && !user.isAnonymous) {
                                        return user;
                                    }
                                }
                            }
                            const sKey = Object.keys(sessionStorage).find(k => k.includes('firebase:authUser'));
                            if (sKey) {
                                const val = sessionStorage.getItem(sKey);
                                if(val) {
                                    const user = JSON.parse(val);
                                    if (user && user.uid && !user.isAnonymous) {
                                        return user;
                                    }
                                }
                            }
                            return "NO_USER"; 
                        } catch(e) {
                            return "SCRIPT_ERROR: " + e.message;
                        }
                    })()
                `;
                    return await newWebview.executeJavaScript(code);
                } catch (e) {
                    return "IPC_ERROR";
                }
            };

            let result = await getUserFromPage();

            if (result && typeof result === 'object' && result.uid && result.email) {
                if (isSigningOut) return;

                const profile = {
                    uid: result.uid,
                    email: result.email,
                    displayName: result.displayName || result.email.split('@')[0],
                    photoURL: result.photoURL
                };

                if (!currentUser || currentUser.uid !== profile.uid) {
                    debugLog.auth('New user detected via storage (Local UI Update only):', profile.email);
                    window.appAPI.saveUserProfile(profile);
                    currentUser = profile;
                    updateProfileUI(profile);
                    await loadAppSettingsFromDisk();
                    await loadDownloadHistoryFromDisk();
                }
            } else if ((result === "NO_USER") && currentUser) {
                if (!isSigningOut) {
                    currentUser = null;
                    resetProfileUI();
                }
            }
        } catch (e) {
            console.error('Auth check error:', e);
        }
    };

    newWebview.addEventListener('did-stop-loading', async () => {
        if (newId === activeWebviewId) hideLoadingScreen();
        for (let i = 0; i < 5; i++) {
            await checkAuthStatus();
            await new Promise(r => setTimeout(r, 1000));
        }
    });

    const pollInterval = setInterval(checkAuthStatus, 3000);
    newWebview.addEventListener('destroyed', () => clearInterval(pollInterval));

    activateTab(newId);
}

// ✅ Load profile on startup
window.onload = async () => {
    minimizeBtn = document.getElementById("minimizeBtn");
    maximizeBtn = document.getElementById("maximizeBtn");
    closeBtn = document.getElementById("closeBtn");
    if (minimizeBtn) minimizeBtn.addEventListener("click", () => window.appAPI.controlWindow('minimize'));
    if (maximizeBtn) maximizeBtn.addEventListener("click", () => window.appAPI.controlWindow('maximize'));
    if (closeBtn) closeBtn.addEventListener("click", () => window.appAPI.controlWindow('close'));

    // 1️⃣ Load settings first (includes theme)
    await loadAppSettingsFromDisk();

    // 2️⃣ Check for existing user profile (Local only)
    await fetchUserProfile();

    // 3️⃣ Load download history
    await loadDownloadHistoryFromDisk();

    // 4️⃣ Create initial tab
    addNewTab(defaultLoadUrl);

    window.appAPI.onWindowStateChange((data) => {
        isMaximized = data.isMaximized;
        updateMaximizeIcon();
    });
    // --- CLEAR HISTORY HANDLER ---
const clearHistoryBtn = document.getElementById('clearHistoryBtn');
if (clearHistoryBtn) {
    clearHistoryBtn.addEventListener('click', async () => {
        const confirmed = await showConfirmDialog(
            "Clear Download History",
            "Are you sure you want to delete all entries from your history? This will only remove history records, not the actual files.",
            "🗑️"
        );
        
        if (confirmed) {
            // 1. Clear history in main process (which deletes the file on disk)
            window.appAPI.clearDownloadHistory();
            
            // 2. Clear local array and UI
            downloadHistory = [];
            
            // 3. Rerender the sidebar to show the "No downloads yet" state
            renderSidebarHistory();
            
            // 4. Hide the badge
            if (dlBadge) dlBadge.style.display = 'none';

            console.log('✅ Download history cleared.');
        }
    });
}
    try {
        const state = await window.appAPI.getInitialWindowState();
        isMaximized = state.isMaximized;
        updateMaximizeIcon();
    } catch (error) {}

    if (webBox) {
        webBox.classList.add('fullwidth');
    }
};

if (window.appAPI && window.appAPI.onOpenUrlInNewTab) {
    window.appAPI.onOpenUrlInNewTab((url) => {
        console.log('📨 Received URL from main process:', url);
        addNewTab(url);
    });
}

// --- TOOLBAR HANDLERS ---
homeBtn.addEventListener("click", () => {
    const webview = getActiveWebview();
    if (webview) webview.loadURL(defaultLoadUrl);
});
backBtn.addEventListener("click", () => {
    const webview = getActiveWebview();
    if (webview) webview.goBack();
});
forwardBtn.addEventListener("click", () => {
    const webview = getActiveWebview();
    if (webview) webview.goForward();
});
reloadBtn.addEventListener("click", () => {
    const webview = getActiveWebview();
    if (webview) webview.reload();
});
profileBtn.addEventListener("click", () => {
    const isVisible = profileDropdown.style.display === "flex";
    profileDropdown.style.display = isVisible ? "none" : "flex";
});

function sendSettings() {
    window.appAPI.updateSettings({
        popupBeforeInstall: togglePopup.checked,
        autoInstall: toggleAuto.checked
    });
}
toggleAuto.addEventListener("change", () => {
    if (toggleAuto.checked) togglePopup.checked = false;
    sendSettings();
});
togglePopup.addEventListener("change", () => {
    if (togglePopup.checked) toggleAuto.checked = false;
    sendSettings();
});
toggleDNS.addEventListener("change", () => {
    window.appAPI.toggleDNS(toggleDNS.checked);
});
// --- TOOLBAR HANDLERS ---
// ... (keep existing code for homeBtn, backBtn, forwardBtn, reloadBtn, profileBtn, toggles) ...
appInfoTabBtn.addEventListener("click", async () => {
    const activeView = getActiveWebview();
    const activeTab = tabs.find(t => t.id === activeWebviewId);
    
    if (!activeView) {
        document.getElementById("infoAppName").innerText = "No Active Tab";
        document.getElementById("infoPublisher").innerText = "N/A";
        document.getElementById("infoVersion").innerText = "N/A";
        document.getElementById("infoPlatform").innerText = "Web Browser";
        document.getElementById("infoAccountStatus").innerText = "N/A";
        document.getElementById("infoInstalledDate").innerText = "N/A";
        document.getElementById("infoStorageSpace").innerText = "0 MB";
        appInfoModal.style.display = "flex";
        return;
    }
    
    try {
        // Get basic info from the active webview
        const url = activeView.getURL();
        const title = activeTab?.tabBtn.querySelector('.tab-title')?.innerText || "Unknown";
        
        // Parse domain from URL
        let domain = "Unknown";
        let publisher = "Unknown";
        try {
            const urlObj = new URL(url);
            domain = urlObj.hostname;
            // Extract main domain (e.g., "naroapp.store" from "www.naroapp.store")
            publisher = domain.replace(/^www\./, '');
        } catch (e) {
            domain = "Local";
            publisher = "Local Application";
        }
        
        // Calculate storage (from cache/cookies)
        let storageSize = "Unknown";
        try {
            const storageData = await activeView.executeJavaScript(`
                (function() {
                    let size = 0;
                    for (let key in localStorage) {
                        if (localStorage.hasOwnProperty(key)) {
                            size += localStorage[key].length + key.length;
                        }
                    }
                    return (size / 1024).toFixed(2); // Convert to KB
                })()
            `);
            storageSize = storageData + " KB";
        } catch (e) {
            storageSize = "N/A";
        }
        
        // Get user account status
        let accountStatus = "Not Logged In";
        if (currentUser && currentUser.email) {
            accountStatus = `Logged in as ${currentUser.email}`;
        }
        
        // Update modal with real data
        document.getElementById("infoAppName").innerText = title || "Unknown Page";
        document.getElementById("infoPublisher").innerText = publisher;
        document.getElementById("infoVersion").innerText = "Web App";
        document.getElementById("infoPlatform").innerText = "Naroapp Browser (Electron)";
        document.getElementById("infoAccountStatus").innerText = accountStatus;
        document.getElementById("infoInstalledDate").innerText = new Date().toLocaleDateString();
        document.getElementById("infoStorageSpace").innerText = storageSize;

        appInfoModal.style.display = "flex";
        
    } catch (e) {
        console.error("Error loading app info:", e);
        document.getElementById("infoAppName").innerText = "Error Loading Info";
        document.getElementById("infoPublisher").innerText = "N/A";
        document.getElementById("infoVersion").innerText = "N/A";
        document.getElementById("infoPlatform").innerText = "N/A";
        document.getElementById("infoAccountStatus").innerText = "N/A";
        document.getElementById("infoInstalledDate").innerText = "N/A";
        document.getElementById("infoStorageSpace").innerText = "N/A";
        appInfoModal.style.display = "flex";
    }
});

// --- DOWNLOAD UI ---
downloadBtn.addEventListener("click", () => {
    const sidebar = document.getElementById('downloadSidebar');
    
    if (sidebar.classList.contains('visible')) {
        sidebar.classList.remove('visible');
        sidebar.classList.add('collapsed');
    } else {
        sidebar.classList.remove('collapsed');
        sidebar.classList.add('visible');
    }
    
    // Hide badge when opened
    if (sidebar.classList.contains('visible')) {
        dlBadge.style.display = "none";
    }
});

// Updated function to handle download progress/completion logic
function createOrUpdateDownloadItem(filename, progress, path, totalSize, date) {
    // Note: We no longer create temporary items in the popover.
    // We update the main history array and re-render the sidebar.

    const downloadDate = date || new Date().toLocaleDateString();
    
    const historyItem = {
        filename: filename,
        path: path || 'N/A',
        size: totalSize || 'N/A',
        date: downloadDate,
        progress: progress,
        isPaused: false // Default state
    };

    const existingIndex = downloadHistory.findIndex(h => h.filename === filename);
    if (existingIndex >= 0) {
        // Preserve paused state if it exists
        if (typeof downloadHistory[existingIndex].isPaused !== 'undefined') {
            historyItem.isPaused = downloadHistory[existingIndex].isPaused;
        }
        // Update existing
        downloadHistory[existingIndex] = { ...downloadHistory[existingIndex], ...historyItem };
    } else {
        // Add new
        downloadHistory.unshift(historyItem);
    }
    
    saveDownloadHistoryToDisk();
    renderSidebarHistory();
}

// Handle Pause/Resume events from main process
window.appAPI.onDownloadPaused((data) => {
    console.log('⏸️ Download paused:', data.filename);
    const index = downloadHistory.findIndex(h => h.filename === data.filename);
    if (index >= 0) {
        downloadHistory[index].isPaused = true;
        renderSidebarHistory();
    }
});

window.appAPI.onDownloadResumed((data) => {
    console.log('▶️ Download resumed:', data.filename);
    const index = downloadHistory.findIndex(h => h.filename === data.filename);
    if (index >= 0) {
        downloadHistory[index].isPaused = false;
        renderSidebarHistory();
    }
});

window.appAPI.onDownloadStart((data) => {
    console.log('🎯 [RENDERER] onDownloadStart triggered!', data);
    const totalMB = (data.totalBytes / (1024 * 1024)).toFixed(2) + ' MB';
    createOrUpdateDownloadItem(data.filename, 0, 'N/A', totalMB);
    dlBadge.style.display = "block";
    
    // Auto open sidebar on download start
    const sidebar = document.getElementById('downloadSidebar');
    if (sidebar && sidebar.classList.contains('collapsed')) {
        sidebar.classList.remove('collapsed');
        sidebar.classList.add('visible');
    }
    
    console.log('✅ Download item created:', data.filename);
});

window.appAPI.onDownloadProgress(throttle((data) => {
    if (data.progress % 10 === 0) {
        console.log('📊 [RENDERER] onDownloadProgress:', data.filename, data.progress + '%');
    }
    const totalMB = (data.totalBytes / (1024 * 1024)).toFixed(2) + ' MB';
    createOrUpdateDownloadItem(data.filename, data.progress, 'N/A', totalMB);
}, 100));

window.appAPI.onDownloadComplete((data) => {
    console.log('🎉 [RENDERER] onDownloadComplete:', data);
    if (data.success) {
        createOrUpdateDownloadItem(data.filename, 100, data.path, data.totalSize);
        addDownloadToHistory(data.filename, data.path, data.totalSize);
        updateProfileStats();
    } else {
        createOrUpdateDownloadItem(data.filename, -1, 'N/A', data.totalSize);
    }
});

async function updateProfileStats() {
    try {
        const historyCount = downloadHistory.length;
        
        // Calculate total storage from download history
        let totalStorageMB = 0;
        downloadHistory.forEach(item => {
            if (item.size && item.size !== 'N/A') {
                // Extract numeric value from size string (e.g., "15.5 MB" -> 15.5)
                const sizeMatch = item.size.match(/[\d.]+/);
                if (sizeMatch) {
                    totalStorageMB += parseFloat(sizeMatch[0]);
                }
            }
        });

        console.log(`📊 Stats Updated: ${historyCount} downloads, ${totalStorageMB.toFixed(2)} MB`);

        // Update the UI elements
        const statsDownloads = document.getElementById('statsDownloads');
        const statsStorage = document.getElementById('statsStorage');
        
        if (statsDownloads) {
            statsDownloads.innerText = historyCount;
        }
        
        if (statsStorage) {
            statsStorage.innerText = totalStorageMB.toFixed(2) + ' MB';
        }
    } catch (e) {
        console.error('Error updating profile stats:', e);
    }
}

// --- INSTALL MODAL ---
window.appAPI.onShowInstallModal((data) => {
    lastDownloadedFile = data.path;
    const guessedAppName = getCleanAppName(data.filename);
    modalFileName.innerText = `Ready to install: ${data.filename}`;
    installModal.style.display = "flex";
    installModal.dataset.appName = guessedAppName;
    installModal.dataset.filename = data.filename;
    installNowBtn.disabled = false;
    installNowBtn.style.opacity = "1";
});

installNowBtn.addEventListener("click", () => {
    installNowBtn.disabled = true;
    installNowBtn.style.opacity = "0.5";
    if (lastDownloadedFile) {
        const activeView = getActiveWebview();
        const appName = installModal.dataset.appName;

        window.appAPI.installApp(lastDownloadedFile);

        updateAppInstallationStatus({
            appName: appName,
            filename: installModal.dataset.filename || "unknown.exe",
            path: lastDownloadedFile,
            status: 'installed',
            size: 'N/A' 
        });

        if (activeView) {
            activeView.send("install-start-signal", {
                type: "INSTALL_INITIATED",
                appName: appName
            });
        }
    }
    setTimeout(() => {
        installModal.style.display = "none";
    }, 200);
});

closeInstallModalBtn.addEventListener("click", () => {
    installModal.style.display = "none";
});
laterBtn.addEventListener("click", () => {
    installModal.style.display = "none";
});

// ... rest of existing code ... (keeping sync, settings, appearance logic)
profileSyncBtn.addEventListener("click", () => {
    syncSettingsModal.style.display = "flex";
});
closeSyncSettings.addEventListener("click", () => {
    syncSettingsModal.style.display = "none";
});

// ✅ FIX: Sync toggle with restart warning
settingSyncToggle.addEventListener("change", async () => {
    const newState = settingSyncToggle.checked;
    const oldState = syncConfig.enabled;
    
    // If no change, do nothing
    if (newState === oldState) return;
    
    // Show custom restart warning
    const restartWarning = document.createElement('div');
    restartWarning.id = 'restartWarning';
    restartWarning.className = 'restart-warning';
    restartWarning.innerHTML = `
        <div class="restart-box">
            <div class="restart-icon">⚠️</div>
            <div class="restart-title">Restart Required</div>
            <div class="restart-message">
                ${newState ? 'Enabling' : 'Disabling'} sync requires switching to ${newState ? 'Firebase' : 'Standard'} mode.<br>
                The app will restart automatically.
            </div>
            <div class="restart-countdown" id="restartCountdown">Auto-restart in 5 seconds...</div>
            <div class="restart-buttons">
                <button class="restart-btn restart-btn-yes" id="restartYes">Restart Now</button>
                <button class="restart-btn restart-btn-no" id="restartNo">Cancel</button>
            </div>
        </div>
    `;
    document.body.appendChild(restartWarning);
    
    let countdown = 5;
    const countdownEl = document.getElementById('restartCountdown');
    const countdownInterval = setInterval(() => {
        countdown--;
        if (countdown > 0) {
            countdownEl.innerText = `Auto-restart in ${countdown} seconds...`;
        } else {
            clearInterval(countdownInterval);
            performRestart();
        }
    }, 1000);
    
    const performRestart = async () => {
        // Save new state
        syncConfig.enabled = newState;
        localStorage.setItem('syncEnabled', newState);
        
        // Update main process settings
        const settings = await window.appAPI.getAppSettings();
        settings.syncEnabled = newState;
        window.appAPI.updateSettings(settings);
        
        // Wait a moment for settings to save
        setTimeout(() => {
            // Restart app
            if (window.appAPI && window.appAPI.restartApp) {
                window.appAPI.restartApp();
            } else {
                window.location.reload();
            }
        }, 500);
    };
    
    document.getElementById('restartYes').addEventListener('click', () => {
        clearInterval(countdownInterval);
        performRestart();
    });
    
    document.getElementById('restartNo').addEventListener('click', () => {
        clearInterval(countdownInterval);
        settingSyncToggle.checked = oldState; // Revert checkbox
        restartWarning.remove();
    });
});

const settingRunAsAdmin = document.getElementById("settingRunAsAdmin");
const settingStartupLaunch = document.getElementById("settingStartupLaunch");
const settingMinimizeToTray = document.getElementById("settingMinimizeToTray");
const settingEcoMode = document.getElementById("settingEcoMode");

async function updateSetting(settingName, value) {
    try {
        const settings = await window.appAPI.getAppSettings();
        settings[settingName] = value;
        window.appAPI.updateSettings(settings);
        console.log(`Updated ${settingName} to ${value}`);
    } catch (e) {
        console.error('Error updating setting:', e);
    }
}
if (settingRunAsAdmin) {
    settingRunAsAdmin.addEventListener('change', () => {
        updateSetting('runAsAdmin', settingRunAsAdmin.checked);
    });
}
if (settingStartupLaunch) {
    settingStartupLaunch.addEventListener('change', () => {
        updateSetting('startupLaunch', settingStartupLaunch.checked);
    });
}
if (settingMinimizeToTray) {
    settingMinimizeToTray.addEventListener('change', () => {
        updateSetting('minimizeToTray', settingMinimizeToTray.checked);
    });
}
if (settingEcoMode) {
    settingEcoMode.addEventListener('change', () => {
        updateSetting('ecoMode', settingEcoMode.checked);
        if (settingEcoMode.checked) {
            document.documentElement.setAttribute('data-eco', 'true');
        } else {
            document.documentElement.removeAttribute('data-eco');
        }
    });
}

// --- APPEARANCE MODAL ---
profileAppearanceBtn.addEventListener('click', () => {
    profileDropdown.style.display = 'none';
    appearanceModal.style.display = 'flex';
});
closeAppearance.addEventListener('click', () => {
    appearanceModal.style.display = 'none';
});
// Add this NEW handler
if (profileAccountSettingsBtn) {
    profileAccountSettingsBtn.addEventListener('click', () => {
        profileDropdown.style.display = 'none';
        accountSettingsModal.style.display = 'flex';
    });
}

if (closeAccountSettings) {
    closeAccountSettings.addEventListener('click', () => {
        accountSettingsModal.style.display = 'none';
    });
}
const profileNotificationsBtn = document.getElementById("profileNotificationsBtn");
if (profileNotificationsBtn) {
    profileNotificationsBtn.addEventListener('click', () => {
        console.log('🔔 Opening Naroapp Aratt.ai profile');
        profileDropdown.style.display = 'none';
        // Open Aratt.ai profile in new tab
        addNewTab("https://naroapp.store/pages/articles.html");
    });
}
document.querySelectorAll('.theme-option').forEach(option => {
    option.addEventListener('click', async () => {
        const themeName = option.dataset.theme;
        applyTheme(themeName);
        const settings = await window.appAPI.getAppSettings();
        settings.theme = themeName;
        window.appAPI.updateSettings(settings);
    });
});

// --- SIGN OUT HANDLER ---
if (profileSignOutBtn) {
    profileSignOutBtn.addEventListener("click", async () => {
        const confirmed = await showConfirmDialog(
            "Sign Out",
            "Are you sure you want to sign out? Your local settings will be preserved.",
            "🚪"
        );
        if (!confirmed) return;

        try {
            isSigningOut = true;
            debugLog.auth("Starting sign out process...");

            // 1. Clear webview storage
            const activeView = getActiveWebview();
            if (activeView) {
                try {
                    await activeView.executeJavaScript(`
                        try {
                            localStorage.clear();
                            sessionStorage.clear();
                            indexedDB.databases && indexedDB.databases().then(dbs => {
                                dbs.forEach(db => indexedDB.deleteDatabase(db.name));
                            });
                            console.log('✅ Webview storage cleared');
                        } catch(e) {
                            console.error('Webview cleanup error:', e);
                        }
                    `);
                } catch (e) {
                    console.error("Failed to clear webview storage:", e);
                }
            }

            // 2. Clear local profile via Main Process
            const result = await window.appAPI.clearUserProfile();
            console.log("Profile clear result:", result);

            // 3. Reset UI
            currentUser = null;
            resetProfileUI();
            profileDropdown.style.display = "none";

            // 4. Redirect to login page in webview
            if (activeView) {
                activeView.loadURL("https://www.naroapp.store/pages/login.html");
            }

            debugLog.auth("✅ Sign out completed");

            setTimeout(() => {
                isSigningOut = false;
            }, 2000);

        } catch (error) {
            console.error("❌ Sign out failed:", error);
            isSigningOut = false;
        }
    });
}
function updateTabScrollIndicators() {
    if (!tabBar) return;
    
    const isScrollable = tabBar.scrollWidth > tabBar.clientWidth;
    const isAtStart = tabBar.scrollLeft <= 0;
    const isAtEnd = tabBar.scrollLeft >= (tabBar.scrollWidth - tabBar.clientWidth - 1);
    
    // Add/remove classes for styling
    tabBar.classList.toggle('scrollable', isScrollable);
    tabBar.classList.toggle('scroll-start', isAtStart);
    tabBar.classList.toggle('scroll-end', isAtEnd);
}

// Update indicators on scroll and tab changes
if (tabBar) {
    tabBar.addEventListener('scroll', updateTabScrollIndicators);
    
    // Also update when tabs are added/removed
    const observer = new MutationObserver(updateTabScrollIndicators);
    observer.observe(tabBar, { childList: true });
    
    // Initial check
    updateTabScrollIndicators();
}

// ==========================================
// COMPLETE IMPLEMENTATION EXAMPLE
// ==========================================
// Place this entire block in your window.onload or after DOM is ready

(function initTabAndDownloadFixes() {
    console.log('🔧 Initializing tab scroll and download bar fixes...');
    
    // Tab bar scroll fix
    const tabBar = document.getElementById('tabBar');
    if (tabBar) {
        tabBar.addEventListener('wheel', (e) => {
            e.preventDefault();
            tabBar.scrollLeft += e.deltaY;
            if (e.deltaX !== 0) {
                tabBar.scrollLeft += e.deltaX;
            }
        }, { passive: false });
        
        console.log('✅ Tab bar mouse scroll enabled');
    }
    
    // Download bar glitch prevention is handled by CSS
    // Just ensure re-renders are throttled (see above)
    
    console.log('✅ Fixes applied successfully');
})();
